"use strict"

var AppDispatcher = require('../dispatcher/AppDispatcher');
var EventEmitter = require('events').EventEmitter;
var SensorActionConstants = require('../constants/SensorActionConstants');
var SensorTypeConstants = require('../constants/SensorTypeConstants');
var GlobalConstants = require('../constants/GlobalConstants');
var FilenameGenerator = require('../helpers/FilenameGenerator');
var UnitsConverter = require('../helpers/UnitsConverter');
var AppActionConstants = require('../constants/AppActionConstants');
var TabConstants = require('../constants/TabConstants');
var Errors = require('../objects/sensor/Errors');
var assign = require('object-assign');

//event names
var SENSOR_CONNECTED_EVENT = 'sensorConnectedEvent';
var SENSOR_DISCONNECTED_EVENT = 'sensorDisconnectedEvent';
var PROFILE_RECEIVED_EVENT = 'profileReceived';
var MEASURE_DATA_RECEIVED_EVENT = 'measureDataReceived';
var TABLE_ITEMS_RECEIVED_EVENT = 'TableItemsReceived';
var CONFIG_CHANGED_EVENT = 'configChangedEvent';
var TAB_CHANGED_EVENT = 'tabChangedEvent';
var FW_VERSION_EVENT = 'fwVersionEvent';
var FLASH_WRITTEN_EVENT = 'flashWrittenEvent';
var FLASH_READ_EVENT = 'flashReadEvent';
var Sensor = require('../objects/sensor/Sensor');
var sensor = new Sensor();

var SensorConfig = require('./../objects/sensor/SensorConfig');
var sensorConfig = new SensorConfig();

var CONTINUOUS_TIMER_INTERVAL = 30; //time between continuous requests


sensor
	.on('connect', function(connInfo){
		SensorStore.getParams();
		SensorStore.emitSensorConnected(connInfo);
	})
	.on('getThreshold', function(data){
		sensorConfig.setThreshold(data);
		SensorStore.emitConfigChanged(sensorConfig);
	})
	.on('getMode', function(value){
		sensorConfig.setMode(value);
		SensorStore.emitConfigChanged(sensorConfig);
	})
	.on('getActiveLines', function(value){
		sensorConfig.setActiveLines(value);
		SensorStore.emitConfigChanged(sensorConfig);
	})
	.on('tableGetItems', function(value){
		SensorStore.emitTableItemsReceived(value);
	})
	.on('error', function(error){
		switch (error.name){
			case Errors.TIMEOUT:
				alertify.error('Connection timeout');
				break;
			case Errors.ALREADY_CONNECTED:
				alertify.log('Already connected');
				break;
			case Errors.NOT_AVAILABLE:
				alertify.error('Sensor not available or already in use');
				break;
			case Errors.TRANSFER_ERROR:
				alertify.error('Data transfer error');
				break;
			default:
				alertify.error('Error in sensor connection');
				break;				
		}
	})
	.on('close', function(){
		SensorStore.connectionInfo = null;
		SensorStore.sensorType = null;
		SensorStore.emitSensorDisconnected();
	})
	.on('getProfile', function(data){
		SensorStore.emitProfileReceived({
			data : data,
			type : GlobalConstants.PROFILE_CMOS
		});
	})
	.on('getProfileNormalized', function(data){
		SensorStore.emitProfileReceived({
			data : data,
			type : GlobalConstants.PROFILE_NORMALIZED
		});
	})
	.on('getProfileNormalizedFiltered', function(data){
		SensorStore.emitProfileReceived({
			data : data,
			type : GlobalConstants.PROFILE_NORMALIZED_FILTERED
		});
	})
	.on('get1Data', function(data){
		SensorStore.emitMeasureDataReceived(data);
	})
	.on('getVersion', function(data){
		SensorStore.emitFwVersion(data);
	})
	.on('flashToRam', function(){
		SensorStore.emitFlashRead();
		SensorStore.getParams(); //after loading flash params to sensor - refresh params in application
	})
	.on('ramToFlash', function(){
		SensorStore.emitFlashWritten();
	})
	.on('normalize', function(success){
		console.log('normalize result', success);
		if (success){
			alertify.success('Normalization OK');
		}else{
			alertify.error('Object in emasuring area - can\'t normalize');
		}
	})
	;
	// .on('readEeprom', function(data){
	// 	writeEepromFile(data, function(err) {
	// 		if (err) {
	// 			console.log(err);
	// 			SensorStore.emitEepromRead(false);
	// 		}else{
	// 			SensorStore.emitEepromRead(true);
	// 		}
	// 	});
	// })
	// .on('writeEeprom', function(success){
	// 	SensorStore.emitEepromWrite(success);
	// });
	


var SensorStore = assign({}, EventEmitter.prototype, {
	connectionInfo : null,

	init: function () {
		//unused
	},

	getParams: function () {
		sensor.getVersion();
		sensor.getLaserPower();
		sensor.getMode();
		sensor.getThreshold();
		sensor.getActiveLines();
		sensor.tableGetItems();
	},

	emitSensorConnected: function(connectionInfo) {this.emit(SENSOR_CONNECTED_EVENT, connectionInfo);},
	addSensorConnectedListener: function(callback) {this.on(SENSOR_CONNECTED_EVENT, callback);},
	removeSensorConnectedListener: function(callback) {this.removeListener(SENSOR_CONNECTED_EVENT, callback);},

	emitSensorDisconnected: function() {this.emit(SENSOR_DISCONNECTED_EVENT);},
	addSensorDisconnectedListener: function(callback) {this.on(SENSOR_DISCONNECTED_EVENT, callback);},
	removeSensorDisconnectedListener: function(callback) {this.removeListener(SENSOR_DISCONNECTED_EVENT, callback);},


	emitMeasureDataReceived: function(data) {this.emit(MEASURE_DATA_RECEIVED_EVENT, data);},
	addMeasureDataReceivedListener: function(callback) {this.on(MEASURE_DATA_RECEIVED_EVENT, callback);},
	removeMeasureDataReceivedListener: function(callback) {this.removeListener(MEASURE_DATA_RECEIVED_EVENT, callback);},

	emitTableItemsReceived: function(data) {this.emit(TABLE_ITEMS_RECEIVED_EVENT, data);},
	addTableItemsReceivedListener: function(callback) {this.on(TABLE_ITEMS_RECEIVED_EVENT, callback);},
	removeTableItemsReceivedListener: function(callback) {this.removeListener(TABLE_ITEMS_RECEIVED_EVENT, callback);},


	emitProfileReceived: function(image) {this.emit(PROFILE_RECEIVED_EVENT, image);},
	addProfileReceivedListener: function(callback) {this.on(PROFILE_RECEIVED_EVENT, callback);},
	removeProfileReceivedListener: function(callback) {this.removeListener(PROFILE_RECEIVED_EVENT, callback);},


	emitConfigChanged: function(value) {this.emit(CONFIG_CHANGED_EVENT, value);},
	addConfigChangedListener: function(callback) {this.on(CONFIG_CHANGED_EVENT, callback);},
	removeConfigChangedListener: function(callback) {this.removeListener(CONFIG_CHANGED_EVENT, callback);},


	emitTabChanged: function(value) {this.emit(TAB_CHANGED_EVENT, value);},
	addTabChangedListener: function(callback) {this.on(TAB_CHANGED_EVENT, callback);},
	removeTabChangedListener: function(callback) {this.removeListener(TAB_CHANGED_EVENT, callback);},


	emitFwVersion: function(callback) {this.emit(FW_VERSION_EVENT, callback); },
	addFwVersionListener: function(callback) {this.on(FW_VERSION_EVENT, callback); },
	removeFwVersionListener: function(callback) {this.removeListener(FW_VERSION_EVENT, callback);},

	emitFlashWritten: function(connectionInfo) {this.emit(FLASH_WRITTEN_EVENT, connectionInfo);},
	addFlashWrittenListener: function(callback) {this.on(FLASH_WRITTEN_EVENT, callback);},
	removeFlashWrittenListener: function(callback) {this.removeListener(FLASH_WRITTEN_EVENT, callback);},

	emitFlashRead: function() {this.emit(FLASH_READ_EVENT);},
	addFlashReadListener: function(callback) {this.on(FLASH_READ_EVENT, callback);},
	removeFlashReadListener: function(callback) {this.removeListener(FLASH_READ_EVENT, callback);},

});

// Register callback to handle all updates
AppDispatcher.register(function(action) {

	switch(action.actionType) {
		

		case SensorActionConstants.GET_SENSOR_PARAMS:
			SensorStore.getParams();
			break;			

		case SensorActionConstants.MODE_CHANGE:
			sensor.setMode(action.value);
			sensorConfig.setMode(action.value);
			SensorStore.emitConfigChanged(sensorConfig);
			break;


		// case SensorActionConstants.CONNECT_USB:
		// 	sensor.connectUsb();
		// 	break;

		case SensorActionConstants.CONNECT_SERIAL:
			var comPort = action.value;
			sensor.connectSerial(comPort, Sensor.DEFAULT_BAUD_RATE);
			break;

		case SensorActionConstants.DISCONNECT:
			sensor.disconnect();
			break;

		case SensorActionConstants.NORMALIZE:
			sensor.normalizationSourceUser();
			sensor.normalize();
			break;

		case SensorActionConstants.FLASH_TO_RAM:
			//load params from flash and retrieve all fresh params
			sensor.flashToRam();
			SensorStore.getParams();
			break;

		case SensorActionConstants.RAM_TO_FLASH:
			//save current ram params to flash
			sensor.ramToFlash();
			break;

		case SensorActionConstants.THRESHOLD_CHANGE:
			sensor.setThreshold(action.value);
			sensorConfig.setThreshold(action.value);
			SensorStore.emitConfigChanged(sensorConfig);
			break;
		
		case SensorActionConstants.TABLE_SAVE_ITEM:
			sensor.tableSaveItem();
			break;
		
		case SensorActionConstants.TABLE_DELETE_ITEM:
			sensor.tableDeleteItem(action.value);
			break;
		
		case SensorActionConstants.TABLE_DELETE_ALL:
			sensor.tableDeleteAll();
			break;
		
		case SensorActionConstants.POWER_OFF:
			sensor.powerOff();
			break;

		case AppActionConstants.TAB_CHANGE:

			switch (action.value) {
				case TabConstants.MEASURE:
					sensor.setContinuous(
						[
							Sensor.CONTINUOUS_MEASUREMENT
						],
						CONTINUOUS_TIMER_INTERVAL
					);
					break;
				case TabConstants.PROFILE:
					sensor.setContinuous(
						[
							Sensor.CONTINUOUS_MEASUREMENT,
							Sensor.CONTINUOUS_PROFILE_CMOS,
							Sensor.CONTINUOUS_PROFILE_NORMALIZED_FILTERED,
						],
						CONTINUOUS_TIMER_INTERVAL
					);
					break;
				case TabConstants.SAMPLING_TABLE:
					sensor.setContinuous(
						[
							Sensor.CONTINUOUS_MEASUREMENT,
							Sensor.CONTINUOUS_SAMPLING_TABLE,
						],
						CONTINUOUS_TIMER_INTERVAL
					);
					break;
				// case TabConstants.STREAM:
				case TabConstants.SENSOR:
					sensor.cancelContinuous();
					SensorStore.getParams();
					break;
				}
				SensorStore.emitTabChanged(action.value);
				break;

		default:
			console.log('Unrecognized action in sensorStore', action.actionType);
			// no op
	}
});

module.exports = SensorStore;
