"use strict";

var SerialPort = nw.require("serialport");
var Buffer = nw.require('buffer').Buffer;
var assign = require('object-assign');
var util = require('util');
var EventEmitter = nw.require('events').EventEmitter;
// var Errors = require('./Errors');
var ConnReqMessage = require('./connReqMessage.js');  
var ConnRespMessage = require('./connRespMessage.js');  


var COMMAND_TIMER = 3;
var DEBUG_CONNECTION = false;
var CONNECTION_TIMEOUT = 10000;


util.inherits(SerialConnection, EventEmitter);


function SerialConnection() {
	var self = this;
	EventEmitter.call(this);

	this._connected = false;
	this._serialPort = null;
	this._tag = 0;
	this._currentMessage = null;
	this._receivingDataBuffer = Buffer.alloc(0);
	this._messageBuffer = [];

	this._messageTimer = null;
};


SerialConnection.prototype._calculateChecksum = function (data) {
	// for n,i in enumerate(ar):
	// 	if n!=1: s = (s+ self.opt_ord(i))&255
	// if s==0:
	// 	s = 1
	// return s

	let	s = 0;
	if (!data || !data.length){
		return s;
	}

	for (var i = 0; i < data.length; i++) {
		if (i!=1){
			s = s + data[i];
		}
	}

	if (s==0){
		s = 1;
	}
	return s;
		
};

SerialConnection.prototype.isBusy = function(message){
	return this._currentMessage != null;
};

SerialConnection.prototype._resetReceiveBuffer = function(){
	this._receivingDataBuffer = Buffer.alloc(0);
};

SerialConnection.prototype._resetMessageBuffer = function(){
	this._messageBuffer = [];
	this._currentMessage = null;
};  

SerialConnection.prototype._debug = function() {
	if (DEBUG_CONNECTION) {
		console.log.apply(console, arguments);
	}
};

SerialConnection.prototype._startMessageTimer = function(){
	// don't start if already running
	if (this._messageTimer){
		return;
	}

	this._messageTimer = setInterval(function(){
		//if not waiting for response and have other message to handle
		if (
			this._messageBuffer.length && !this._currentMessage
		){
			this._currentMessage = this._messageBuffer.shift();
			this._executeCommand(this._currentMessage);
		}else{
			this._debug('Can\'t execute!! MeesageBufferLength: ' + this._messageBuffer.length + ', currentMessage: ' + (this._currentMessage ? this._currentMessage.getName() : null));
		}
	}.bind(this), COMMAND_TIMER);
};

SerialConnection.prototype._stopMessageTimer = function(){
	clearInterval(this._messageTimer);
	this._messageTimer = null;
};

SerialConnection.prototype.sendMessage = function(message){	
	//not busy or blocking message -> add to buffer
	if (!this._connected)
		return false;

	//if (!this._currentMessage || message.isBlocking()) {
		this._messageBuffer.push(message);
		this._debug('Adding command to buffer: ' + message.getName() + ', blocking: '  + message.getBlocking());
		this._debug('MeesageBufferLength: ' + this._messageBuffer.length);
	// }else{
	// 	this._debug('Not-blocking message: ' + message.getName() + ' omitted cause other command being executed: ' + this._currentMessage.getName());
	// }
};

SerialConnection.prototype.isBusy = function(){
	return this._currentMessage !== null || this._messageBuffer.length != 0;
};

SerialConnection.prototype._executeCommand = function(message){
	// if cmd == self.PT_SYNC:
	// 	tag = 0
	// elif tag is None:
	// 	self.gtag = (self.gtag+1) & 0xffff
	// 	tag = self.gtag
	// dt_spec = "BBHH"+("Hh"[int(signed)])
	// ar = struct.pack(dt_spec,cmd,0,tag,addr,data)
	// if cmd != self.PT_SYNC:
	// 	cs = self._cmp_csum(ar)	#calculate CHECKSUM
	// 	ar = struct.pack(dt_spec,cmd,cs,tag,addr,data)
	// if self.debug:
	// 	print ("Cmd %d tag %d addr 0x%X data 0x%X"%(cmd,tag,addr,data))
	// self.ser.write(ar)
	// return self._portb_recv(signed=signed)
	
	this._currentMessage = message;

	var connReqMessage = message.getConnReqMessage();	
	if (connReqMessage.getCommandCode() == ConnReqMessage.COMMAND_SYNC){
		message.setTag(0);
		connReqMessage.setTag(0);
	}else if (typeof message.getTag() !== "number"){
		this._tag = (this._tag+1) & 0xffff;
		message.setTag(this._tag);
		connReqMessage.setTag(this._tag);
	}

	var dataToSend = new Uint8Array(8);

	dataToSend[0] = connReqMessage.getCommandCode();

	dataToSend[1] = 0;
	dataToSend[2] = connReqMessage.getTag() % 256;
	dataToSend[3] = connReqMessage.getTag() / 256;
	dataToSend[4] = connReqMessage.getAddress() % 256;
	dataToSend[5] = connReqMessage.getAddress() / 256;
	dataToSend[6] = connReqMessage.getData() % 256;
	dataToSend[7] = connReqMessage.getData() / 256;

	if (connReqMessage.getCommandCode() != ConnReqMessage.COMMAND_SYNC){
		dataToSend[1] = this._calculateChecksum(dataToSend);
	}

	this._debug("Executing");
	this._debug(connReqMessage.toLog());
	this._debug(dataToSend);

	this._serialPort.write(dataToSend);
};

//public functions
SerialConnection.prototype.connect = function(comPort, baud){
	var self = this;

	this._serialPort = new SerialPort(comPort, {
		baudRate: baud
	});

	//bind events on port
	this._serialPort
		.on('open', function(data) {			
			this._connected = true;
			this._startMessageTimer();
			this.emit('connect', this._serialPort.path);
		}.bind(this))
		.on('error', function(error) {
			console.log(error);
			this.disconnect();
			//emit error
			// var errorToEmit = {
			// 	originalError : error
			// };
			// switch (error.errno){
			// 	case 'ETIMEDOUT':
			// 		errorToEmit.name = Errors.TIMEOUT;
			// 		break;
			// 	case 'EISCONN':
			// 		errorToEmit.name = Errors.ALREADY_CONNECTED;
			// 		break;
			// 	case 'EADDRINUSE':
			// 		errorToEmit.name = Errors.NOT_AVAILABLE;
			// 		break;
			// 	default:
			// 		errorToEmit.name = Errors.UNSPECIFIED;
			// 		break;				
			// }
			this.emit('error', error);
		}.bind(this))
		.on('close', function(error) {
			this._connected = false;
			
			//reset buffers and state vars
			this._stopMessageTimer();
			this._resetReceiveBuffer();
			this._currentMessage = null;

			this.emit('close', error);
		}.bind(this))
		.on('data', function(data) {
			//concat data to receiving buffer
			this._receivingDataBuffer = Buffer.concat([this._receivingDataBuffer, data]); 
			
			//we have data for unknown message, throw error and thrash data
			if (!this._currentMessage) {
				this._resetReceiveBuffer();
				console.log('Unknown message, data:', data);
				throw "Unknown message";
				return;
			}

			// ss = self.ser.read(6) # read header
			// rsum = self._cmp_csum(ss)
			// ret,csum,tag,cnt = struct.unpack("BBHH",ss)   #decode header
			// rdat = self.ser.read(2*cnt)  #now read number of data followed (2byte datas)
			// odat = [struct.unpack("Hh"[int(signed)],rdat[i*2:i*2+2])[0] for i in range(cnt)]
			// if self.debug:
			// 	print ("%6s S=%d(%d) T=%d C=%d"%(self._retmap.get(ret,str(ret)),
			// 	csum,rsum,tag,cnt),odat)
			// if ret not in (1,10,11) and exc:
			// 	raise Exception("Portable sent err "+self._retmap[ret])
			// return odat
			if (this._receivingDataBuffer.legth < 6){
				this._debug('Waiting for full header');
				return;
			}
			
			//decode header
			var responseCode = this._receivingDataBuffer[0];
			var receivedCheckSum = this._receivingDataBuffer[1];
			var tag = this._receivingDataBuffer[2] + this._receivingDataBuffer[3] * 256;
			var responseSize = (this._receivingDataBuffer[4] + this._receivingDataBuffer[5] * 256) * 2;
			var fullResponseSize = responseSize + 6;

			if (this._receivingDataBuffer.length < fullResponseSize){
				this._debug('Waiting for full data');
				return;	
			}
			
			//if more data 
			if (this._receivingDataBuffer.length > fullResponseSize){
				this._debug('Too much data');
			}

			this._debug('Data received');
			this._debug(this._receivingDataBuffer);

			var connRespMessage = new ConnRespMessage();
			connRespMessage.setResponseCode(responseCode);
			connRespMessage.setCheckSum(receivedCheckSum);
			connRespMessage.setTag(tag);
			connRespMessage.setResponseSize(responseSize);

			// var outData = [];
			// for (var i = 6; i < this._receivingDataBuffer.length; i+=2) {
			// 	outData.push(this._receivingDataBuffer.readUInt16LE(i));
			// }
			// connRespMessage.setData(outData);
			connRespMessage.setData(this._receivingDataBuffer.slice(6));

			this._currentMessage.setConnRespMessage(connRespMessage);

			this.emit('data', this._currentMessage);
			this._currentMessage = null;
			this._resetReceiveBuffer();
		}.bind(this));
};

SerialConnection.prototype.disconnect = function(){
	if (!this._connected){
		this.emit('close');
		return;
	}
	this._serialPort.close();
};




//exposure
module.exports = SerialConnection;