"use strict";

var EventEmitter = require('events').EventEmitter;
var assign = require('object-assign');
var net = require('net');
var util = require('util');

var ConnReqMessage = require('./connReqMessage.js');  
var ConnRespMessage = require('./connRespMessage.js');  
var CommandSet = require('./commandset.js');  
var commandSet = new CommandSet();

//stream timer interval if custom value will not be passed as argument
var DEFAULT_CONTINUOUS_TIMER_INTERVAL = 20;

//continuous data streams
var continousTimer;
var continuousDisable = false;

var PIXEL_SIZE = 0.4375;
var THRESHOLD_SUBPIXEL_MULTIPLIER = 8;



util.inherits(Sensor, EventEmitter);

function Sensor() {
	EventEmitter.call(this);

	var self = this;
	var SerialConnection = require('./serialConnection');

	this._activeLines = {
		from : 20,
		to : 2028
	};
	this.connection = new SerialConnection();
	this.connection
		.on('connect', function(data) {
			self.emit('connect', {text:data});
		})
		.on('timeout', function() {
			self.emit('timeout');
		})
		.on('error', function(error) {
			self.emit('error', error);
		})
		.on('close', function(error) {
			self.emit('close', error);
		})
		.on('data', function(message) {

			//handle data depending on command
			switch (message.getName()) {
				case 'get1Data':
				{
					let dataTmp = (message.getConnRespMessage()).getData();
					let result = [
						PIXEL_SIZE * (dataTmp[0] + dataTmp[1] * 256),
						PIXEL_SIZE * (dataTmp[2] + dataTmp[3] * 256),
						PIXEL_SIZE * (dataTmp[4] + dataTmp[5] * 256),
						PIXEL_SIZE * (dataTmp[6] + dataTmp[7] * 256),
						PIXEL_SIZE * (dataTmp[8] + dataTmp[9] * 256),
						PIXEL_SIZE * (dataTmp[10] + dataTmp[11] * 256)
					];
					// console.log(message.getName() + ' OK ---', result);
					self.emit(message.getName(), result);
					break;
				}

				case 'getProfile':
				{
					let dataTmp = (message.getConnRespMessage()).getData();
					let profile = [];
					for (var i = 0; i < dataTmp.length; i+=2) {
						profile.push(dataTmp[i] + dataTmp[i+1]*256);
					}
					// console.log(message.getName() + ' OK ---', profile.length);
					self.emit(message.getName(), profile);
					break;
				}

				case 'getProfileNormalized':
				case 'getProfileNormalizedFiltered':
				{
					let divider = message.getName() == 'getProfileNormalizedFiltered' ? 8 : 1;
					let dataTmp = (message.getConnRespMessage()).getData();
					let profile = [];
					for (var i = 0; i < 26; i++) {
						profile.push(0);
					}
					for (var i = 0; i < dataTmp.length; i+=2) {
						profile.push(Math.round((dataTmp[i] + dataTmp[i+1]*256) / divider));
					}
					for (var i = 0; i < 26; i++) {
						profile.push(0);
					}
					// console.log(message.getName() + ' OK ---', profile.length);
					self.emit(message.getName(), profile);
					break;
				}

				case 'getLaserPower':
				case 'tableGetCount':
				{
					let dataTmp = (message.getConnRespMessage()).getData();
					self._messagePromiseResolve(message, dataTmp[0]);
					self.emit(message.getName(), dataTmp[0]);
					// console.log(message.getName() + ' OK ---', dataTmp[0]);
					break;
				}

				case 'tableGetItems':
				{	
					let res = [];
					let dataTmp = (message.getConnRespMessage()).getData();
					let timestamp;
					let timezoneOffset = (new Date()).getTimezoneOffset() * 60;
					for (var i = 0; i < dataTmp.length; i+=6) {
						timestamp = ((dataTmp[i+2] << 0) + (dataTmp[i + 3] << 8) + (dataTmp[i + 4] << 16)  + (dataTmp[i + 5] << 24));
						timestamp += timezoneOffset;
						res.push({
							value : Math.round((dataTmp[i] + (dataTmp[i+1] << 8)) * PIXEL_SIZE),
							time : new Date(timestamp * 1000)
						});
					}					
					self.emit(message.getName(), res);
					break;
				}

				case 'getThreshold':
				{
					let dataTmp = (message.getConnRespMessage()).getData();
					let threshold = (dataTmp[0] + dataTmp[1] * 256) / THRESHOLD_SUBPIXEL_MULTIPLIER ;
					// console.log(message.getName() + ' OK ---', threshold);
					self.emit(message.getName(), threshold);
					break;
				}

				case 'getActiveLines':
				{
					let dataTmp = (message.getConnRespMessage()).getData();
					let from = (dataTmp[0] + dataTmp[1] * 256);
					let to = (dataTmp[2] + dataTmp[3] * 256);
					let result = {
						from,
						to
					};
					this._activeLines = result;
					console.log(message.getName() + ' OK ---', result);
					self.emit(message.getName(), result);
					break;
				}

				case 'isObject':
				{
					let dataTmp = (message.getConnRespMessage()).getData();
					let result = dataTmp[0] > 0;

					// console.log(message.getName() + ' OK ---', dataTmp);
					self._messagePromiseResolve(message, result);
					self.emit(message.getName(), result);
					break;
				}

				//one byte params - low byte
				case 'getMode':
				{
					let dataTmp = (message.getConnRespMessage()).getData();
					let mode = dataTmp[0];
					console.log(message.getName() + ' OK ---', mode);
					self.emit(message.getName(), mode);
					break;
				}

				case 'flashToRam':
				case 'ramToFlash':
				case 'tableDeleteAll':
				case 'tableDeleteItem':
				case 'tableSaveItem':
				{
					//console.log(message.getName() + ' OK ---', (message.getConnRespMessage()).getData());
					self.emit(message.getName());
					break;
				}

				case 'normalize':
				{
					// console.log(message.getName() + ' OK ---', (message.getConnRespMessage()).getData());
					self.emit(message.getName(), true);
					break;
				}

				case 'getVersion':
				{
					let dataTmp = (message.getConnRespMessage()).getData();
					let svnRevision = dataTmp[0] + dataTmp[1]*256;
					let hwRevision = dataTmp[10] + dataTmp[11]*256;
					let name = "";
					for (var i = 2; i < 10; i++) {
						name += String.fromCharCode(dataTmp[i]); 
					}
					let result = {
						name : name,
						svnRevision : svnRevision,
						hwRevision : hwRevision
					}
					self.emit(message.getName(), result);
					break;
				}

				default:{
					//console.log('Undefined data handling in sensor onData handler:' + message.getName());
				}
			}
		}.bind(this));
};

Sensor.DEFAULT_BAUD_RATE = 115200;

//constants for continuous streams
Sensor.CONTINUOUS_MEASUREMENT = 'continuousMeasurement';
Sensor.CONTINUOUS_SAMPLING_TABLE = 'continuousSamplingTable';
Sensor.CONTINUOUS_PROFILE_CMOS = 'continuousProfileCmos';
Sensor.CONTINUOUS_PROFILE_NORMALIZED = 'continuousProfileNormalized';
Sensor.CONTINUOUS_PROFILE_NORMALIZED_FILTERED = 'continuousProfileNormalizedFiltered';

Sensor.prototype._messagePromiseResolve = function (message, result) {
	if (message.getPromiseResolver()){
		let func = message.getPromiseResolver();
		func(result);
	}
};

Sensor.prototype._reverseBuffer = function (buffer) {
	var tmp = Buffer.alloc(buffer.length);
	for (var i = 0; i < buffer.length; i++) {
		tmp[buffer.length - i - 1] = buffer[i];
	}
	return tmp;
};

Sensor.prototype._parseData = function (buffer) {
	var result = [];
	var reversed = this._reverseBuffer(buffer);
	for (var i = 0; i < buffer.length / DATA_FRAME_LEN; i++) {
		result.push(parseFloat(reversed.readUIntLE(i * DATA_FRAME_LEN, 2)));
	};
	return result;
};

//public functions
Sensor.prototype.connectSerial = function(port, baud){
	this.connection.connect(port, baud);
};

Sensor.prototype.disconnect = function(){
	this.connection.disconnect();
};


/**
 * Starts periodic requests for data
 * @param {array} dataToStream Array of constants CONTINUOUS_MEASUREMENT/CONTINUOUS_PROFILE
 */
Sensor.prototype.setContinuous = function (dataToStream, continousTimerInterval) {
	console.log('setting continuous', dataToStream);

	//cancel previous stream
	this.cancelContinuous();

	var self = this;
	var nextDataToStreamIndex = 0;

	continousTimerInterval = typeof continousTimerInterval === 'number' ? continousTimerInterval : DEFAULT_CONTINUOUS_TIMER_INTERVAL;

	if (dataToStream && dataToStream.length){
		continousTimer = setInterval(function(){
			if (continuousDisable){
				return;
			}
			//wait until not busy and do request for data			
			if (!self.connection.isBusy()) {
				switch (dataToStream[nextDataToStreamIndex]){
					case Sensor.CONTINUOUS_MEASUREMENT:
						self.get1Data();
						break;
					case Sensor.CONTINUOUS_PROFILE_CMOS:
						self.getProfile();
						break;
					case Sensor.CONTINUOUS_PROFILE_NORMALIZED:
						self.getProfileNormalized();
						break;
					case Sensor.CONTINUOUS_PROFILE_NORMALIZED_FILTERED:
						self.getProfileNormalizedFiltered();
						break;
					case Sensor.CONTINUOUS_SAMPLING_TABLE:
						self.tableGetItems();
						break;
					default:
						console.log('unknown stream type', dataToStream[nextDataToStreamIndex]);
						break;
				}

				//next request for next data
				if (nextDataToStreamIndex === dataToStream.length - 1){
					nextDataToStreamIndex = 0;
				}else{
					nextDataToStreamIndex++;
				}
			}

		}, continousTimerInterval);
	}
};

/**
 * Ends periodic request for data
 */
Sensor.prototype.cancelContinuous = function () {
	clearInterval(continousTimer);
	continousTimer = null;
};

Sensor.prototype.disableContinuous = function () {
	continuousDisable = true;		
};

Sensor.prototype.enableContinuous = function () {
	continuousDisable = false;
};

Sensor.prototype.enableContinuous = function () {
	continuousDisable = false;
};


Sensor.prototype.get1Data = function(){
	this.connection.sendMessage(commandSet.get('get1Data'));
};
Sensor.prototype.normalizationSourceUser = function(){
	this.connection.sendMessage(commandSet.get('normalizationSourceUser'));
};
Sensor.prototype.normalize = function(){
	(new Promise((resolver, rejecter) => {				
		var message = commandSet.get('isObject')
		message.setPromiseResolver(resolver);
		message.setPromiseRejecter(rejecter);
		this.connection.sendMessage(message);
	})).then((isObject) => {
		console.log('----------', isObject);
		if (isObject){
			console.log('not sending');
			this.emit('normalize', false)
		}else{
			console.log('sending');
			this.connection.sendMessage(commandSet.get('normalize'));
		}
	});
};
Sensor.prototype.isObject = function(){
	this.connection.sendMessage(commandSet.get('isObject'));
};
Sensor.prototype.ramToFlash = function(){
	this.connection.sendMessage(commandSet.get('ramToFlash'));
};
Sensor.prototype.flashToRam = function(){
	this.connection.sendMessage(commandSet.get('flashToRam'));
};
Sensor.prototype.getProfile = function(){
	this.connection.sendMessage(commandSet.get('getProfile'));
};
Sensor.prototype.getThreshold = function(){
	this.connection.sendMessage(commandSet.get('getThreshold'));
};
Sensor.prototype.getMode = function(){
	this.connection.sendMessage(commandSet.get('getMode'));
};
Sensor.prototype.getLaserPower = function(){
	this.connection.sendMessage(commandSet.get('getLaserPower'));
};
Sensor.prototype.getVersion = function(){
	this.connection.sendMessage(commandSet.get('getVersion'));
};
Sensor.prototype.getActiveLines = function(){
	this.connection.sendMessage(commandSet.get('getActiveLines'));
};
Sensor.prototype.isObject = function(){
	this.connection.sendMessage(commandSet.get('isObject'));
};
Sensor.prototype.tableGetCount = function(){
	this.connection.sendMessage(commandSet.get('tableGetCount'));
};
Sensor.prototype.tableDeleteAll = function(){
	this.connection.sendMessage(commandSet.get('tableDeleteAll'));
};
Sensor.prototype.tableSaveItem = function(){
	this.connection.sendMessage(commandSet.get('tableSaveItem'));
};
Sensor.prototype.tableDeleteItem = function(index){
	this.connection.sendMessage(commandSet.get('tableDeleteItem', {value : index}));
};
Sensor.prototype.tableGetItems = function(){
	(new Promise((resolver, rejecter) => {				
		let message = commandSet.get('tableGetCount')
		message.setPromiseResolver(resolver);
		message.setPromiseRejecter(rejecter);
		this.connection.sendMessage(message);
	})).then((itemsCount) => {
		let msg = commandSet.get('tableGetItems');
		(msg.getConnReqMessage()).setData(itemsCount * 3);
		this.connection.sendMessage(msg);
	});
};
Sensor.prototype.setLaserOn = function(){
	this.connection.sendMessage(commandSet.get('setLaserOn'));
};
Sensor.prototype.setLaserOff = function(){
	this.connection.sendMessage(commandSet.get('setLaserOff'));
};
Sensor.prototype.getProfileNormalized = function(){
	this.connection.sendMessage(commandSet.get('getProfileNormalized'));
};
Sensor.prototype.getProfileNormalizedFiltered = function(){
	this.connection.sendMessage(commandSet.get('getProfileNormalizedFiltered'));
};
Sensor.prototype.setMode = function(value){
	this.connection.sendMessage(commandSet.get('setMode', {value : value}));
};
Sensor.prototype.setThreshold = function(value){
	this.connection.sendMessage(commandSet.get('setThreshold', {value : value * THRESHOLD_SUBPIXEL_MULTIPLIER  }));
};
Sensor.prototype.powerOff = function(){
	this.connection.sendMessage(commandSet.get('powerOff'));
};


//exposure
module.exports = Sensor;
