"use strict";

var assign = require('object-assign');
var util = require('util');
var EventEmitter = require('events').EventEmitter;
var Buffer = nw.require('buffer').Buffer;
var Errors = require('./errors');
var usb = nw.require('usb');

var COMMAND_TIMER = 3;
var DEBUG_CONNECTION = false;
var CONNECTION_TIMEOUT = 10000;

var VENDOR_ID = 1204;
var PRODUCT_ID = 4099;

var STREAM_MSG_NAME = 'stream';
var STREAM_STOP_MSG_NAME = 'streamStop';


util.inherits(UsbConnection, EventEmitter);


function UsbConnection() {
	EventEmitter.call(this);

	this._expectedResponse = false;
	this._messageBuffer = [];
	this._messageTimer = null;
	this._streamStop = false;


	this._device = null;
	this._in = null; //endpoints
	this._out = null; //endpoints
	
	this._connected = true;

	//timer to handle messages
	this._messageTimer = null;

};

UsbConnection.prototype._endMessageLoop = function(){
	clearInterval(this._messageTimer);
	this._messageTimer = null;

	this._messageBuffer = [];
	this._expectedResponse = false;
};

UsbConnection.prototype._debug = function() {
	if (DEBUG_CONNECTION) {
		console.log.apply(console, arguments);
	}
};


/**
 * Run function asynchronically with optional delay and optional args
 */
UsbConnection.prototype._runAsync = function(func, delay){
	var args = Array.prototype.slice.call(arguments);
	args = args.slice(2); //all remaining args

	setTimeout(function () {
		func.apply(this, args);
	}.bind(this), ((delay) ? delay : 0));
};


UsbConnection.prototype._executeCommand = function(message){
	this._debug('Executiong: ', message.getName());
	
	//data to send
	var commandData = message.getCommandToSend();
	var data = message.getData();

	//lengths
	var cmdLen = (commandData && commandData.length) ? commandData.length : 0;
	var dataLen = (data && data.length) ? data.length : 0;

	//buffer to write in socket
	var dataToSend = Buffer.alloc(cmdLen + dataLen);
	
	//copy command data
	for (var i = 0; i < cmdLen; i++) {
		dataToSend[i] = commandData[i];
	};

	//copy message data
	for (var i = 0; i < dataLen; i++) {
		dataToSend[message.getCommandToSend().length + i] = data[i];
	}

	this._out.transfer(dataToSend, function(err){
		if (err) {
			err.name = Errors.TRANSFER_ERROR;
			this._runAsync(this.emit, 0, 'error', err);
			this._runAsync(this.disconnect, 0, 'Connection error while sending data');
		}
	}.bind(this));

	if (message.getResponseSize()) {
		this._in.transfer(message.getResponseSize(), function (err, data) {
			if (err) {
				err.name = Errors.TRANSFER_ERROR;
				this._runAsync(this.emit, 0, 'error', err);
				this._runAsync(this.disconnect, 0, 'Connection error while receiving data');
				return;
			}

			this._debug('Executed and received: ' + message.getName());

			message.setResponse(data);
			this.emit('data', message);
			
			this._expectedResponse = false;
		}.bind(this));
	}
	
};

//public functions
UsbConnection.prototype.connect = function(){
	var devs = usb.getDeviceList();

	this._device = usb.findByIds(VENDOR_ID, PRODUCT_ID);

	if (this._device) {
		try {
			this._device.open()

			this._device.interfaces[0].claim()
			this._in = this._device.interfaces[0].endpoints[1]; //endpoints
			this._out = this._device.interfaces[0].endpoints[0]; //endpoints

			// this._in.timeout = 200;
			// this._out.timeout = 200;

			this._in.on('timeout', function () {
				this._runAsync(this.emit, 0, 'error', 'Connection timeout error');
				this._runAsync(this.disconnect, 0, 'Connection timeout error');
			}.bind(this));

			this._out.on('timeout', function () {
				this._runAsync(this.emit, 0, 'error', 'Connection timeout error');
				this._runAsync(this.disconnect, 0, 'Connection timeout error');
			}.bind(this));

			this._in.on('error', function () {
				this._runAsync(this.emit, 0, 'error', 'Connection endpoints(in) error');
				this._runAsync(this.disconnect, 0, 'Connection endpoints(in) error');
			}.bind(this));

			this._out.on('error', function () {
				this._runAsync(this.emit, 0, 'error', 'Connection endpoints(out) error');				
				this._runAsync(this.disconnect, 0, 'Connection endpoints(out) error');
			}.bind(this));

			this._connected = true;

			//loop to execute messages
			this._messageTimer = setInterval(function(){
				//if not waiting for response and have other message to handle
				if (this._messageBuffer.length && (!this._expectedResponse || this._expectedResponse === 'stream')){
					var toExecute = this._messageBuffer.shift();
					//if no response expected, don't wait for data after execution
					if (!toExecute.getResponseSize()){
						this._expectedResponse = false;
					}else{
						this._expectedResponse = toExecute.getResponseSize();
					}
					this._executeCommand(toExecute);
				}else{
					this._debug('Can\'t execute - busy!! MeesageBufferLength: ' + this._messageBuffer.length);
				}
			}.bind(this), COMMAND_TIMER);

			this.emit('connect', 'USB');
		}catch(e){
			this._runAsync(this.emit, 0, 'error', e);
			this._runAsync(this.emit, 0, 'close');
		}
	}else{
		var e = {};
		e.name = Errors.NOT_AVAILABLE;

		this._runAsync(this.emit, 0, 'error', e);
		this._runAsync(this.emit, 0, 'close', e);
	}
};

/**
 * Disconnect device, opionally with error message
 */
UsbConnection.prototype.disconnect = function(error){
	//reset buffers and state vars, cancel message loop
	this._endMessageLoop();

	//release device after short break to finish requests
	this._runAsync(function (argument) {
		this._device.reset();
		this.emit('close', error);
	});
	
};


/**
 * Busy with sending message and waiting for response?
 */
UsbConnection.prototype.isBusy = function(){
	return Boolean(this._expectedResponse);
};

/**
 * Public function for sending message - if not blocking and busy - throw away
 */
UsbConnection.prototype.sendMessage = function(message){	
	//not busy or blocking message -> add to buffer
	if (!this._connected)
		return false;

	//can execute only streamStop while receiving stream
	if (this._expectedResponse && this._expectedResponse === STREAM_MSG_NAME) {
		if (message.getName() !== STREAM_STOP_MSG_NAME) {
			return false;
		}
	}

	if (!this._expectedResponse || message.isBlocking()) {
		this._messageBuffer.push(message);
		this._debug('Adding command to buffer: ' + message.getName() + ', blocking: '  + message.getBlocking());
		this._debug('MeesageBufferLength: ' + this._messageBuffer.length);
	}else{
		this._debug('Not-blocking message: ' + message.getName() + ' omitted cause other command being executed');
	}
};

//exposure
module.exports = UsbConnection;