"use strict";

var assign = require('object-assign');
var util = require('util');
var EventEmitter = require('events').EventEmitter;
var Buffer = nw.require('buffer').Buffer;
var Errors = require('./errors');
var net = nw.require('net');

var COMMAND_TIMER = 3;
var DEBUG_CONNECTION = false;
var CONNECTION_TIMEOUT = 10000;

var STREAM_MSG_NAME = 'stream';
var STREAM_STOP_MSG_NAME = 'streamStop';


util.inherits(EthernetConnection, EventEmitter);


function EthernetConnection() {
	var self = this;
	EventEmitter.call(this);

	this._sensorSocket = false;
	this._connected = false;
	this._currentMessage = null;
	this._messageBuffer = [];
	this._receivingDataBuffer = Buffer.alloc(0);
	this._messageTimer = null;
	this._streamStop = false;
};

EthernetConnection.prototype._resetReceiveBuffer = function(){
	this._receivingDataBuffer = Buffer.alloc(0);
};

EthernetConnection.prototype._resetMessageBuffer = function(){
	this._messageBuffer = [];
	this._currentMessage = null;
};  

EthernetConnection.prototype._endMessageTimer = function(){
	clearInterval(this._messageTimer);
	this._messageTimer = null;
};

EthernetConnection.prototype._debug = function() {
	if (DEBUG_CONNECTION) {
		console.log.apply(console, arguments);
	}
};


EthernetConnection.prototype._executeCommand = function(message){
	this._debug('Executiong: ', message.getName());
	
	//data to send
	var commandData = message.getCommandToSend();
	var data = message.getData();

	//lengths
	var cmdLen = (commandData && commandData.length) ? commandData.length : 0;
	var dataLen = (data && data.length) ? data.length : 0;

	//buffer to write in socket
	var dataToSend = Buffer.alloc(cmdLen + dataLen);
	
	//copy command data
	for (var i = 0; i < cmdLen; i++) {
		dataToSend[i] = commandData[i];
	};

	//copy message data
	for (var i = 0; i < dataLen; i++) {
		dataToSend[message.getCommandToSend().length + i] = data[i];
	}

	this._sensorSocket.write(dataToSend);
};

//public functions
EthernetConnection.prototype.connect = function(port, ip){
	var self = this;

	this._sensorSocket = new net.Socket();
	this._sensorSocket
		.on('connect', function(data) {
			self._connected = true;

			//timer to handle messages
			self._messageTimer = setInterval(function(){
				//if not waiting for response and have other message to handle
				if (self._messageBuffer.length && (!self._currentMessage || self._currentMessage.getResponseSize() === 'stream')){
					var toExecute = self._messageBuffer.shift();
					//if no response expected, don't wait for data after execution
					if (!toExecute.getResponseSize()){
						self._currentMessage = null;
					}else{
						self._currentMessage = toExecute;
					}
					self._executeCommand(toExecute);
				}else{
					self._debug('Can\'t execute!! MeesageBufferLength: ' + self._messageBuffer.length + ', currentMessage: ' + (self._currentMessage ? self._currentMessage.getName() : null));
				}
			}, COMMAND_TIMER);

			self.emit('connect', self._sensorSocket.remoteAddress);
		})
		.on('timeout', function() {

			this._sensorSocket._destroy();

			var errorToEmit = {
				originalError : null,
				name : Errors.TIMEOUT
			};

			self.emit('error', errorToEmit);
		})
		.on('error', function(error) {
			//emit error
			var errorToEmit = {
				originalError : error
			};
			switch (error.errno){
				case 'ETIMEDOUT':
					errorToEmit.name = Errors.TIMEOUT;
					break;
				case 'EISCONN':
					errorToEmit.name = Errors.ALREADY_CONNECTED;
					break;
				case 'EADDRINUSE':
					errorToEmit.name = Errors.NOT_AVAILABLE;
					break;
				default:
					errorToEmit.name = Errors.UNSPECIFIED;
					break;				
			}
			self.emit('error', errorToEmit);
		})
		.on('close', function(error) {
			self._connected = false;
			self._sensorSocket = null;

			self.emit('close', error);
		})
		.on('data', function(data) {

			//concat data to receiving buffer
			self._receivingDataBuffer = Buffer.concat([self._receivingDataBuffer, data]); 
			
			//we have data for unknown message, throw error and thrash data
			if (!self._currentMessage) {
				self._resetReceiveBuffer();
				self._resetMessageBuffer();
				console.log('Unknown message, data:', data);
				//throw "Unknown message";
				return;
			}

			if (self._currentMessage.getResponseSize() === 'stream'){
				self._debug('Got stream data: ' + self._currentMessage.getName());

				self._currentMessage.setResponse(self._receivingDataBuffer);
				self.emit('data', self._currentMessage);
				self._resetReceiveBuffer();
				//not reset current message - wait until other stream data
				//self._currentMessage = null; 
			}
			//expected data length received
			else if (self._currentMessage.getResponseSize() === self._receivingDataBuffer.length){
				self._debug('Executed and received: ' + self._currentMessage.getName());

				self._currentMessage.setResponse(self._receivingDataBuffer);
				self.emit('data', self._currentMessage);
				self._resetReceiveBuffer();
				self._currentMessage = null;
			}
			//need to handle more data than expected
			else if(self._currentMessage.getResponseSize() < self._receivingDataBuffer.length) {
				self._debug('slicing receive buffer');

				//emit only expected data and left the rest for next message
				var bufferToEmit = Buffer.alloc(self._currentMessage.getResponseSize());
				self._receivingDataBuffer.copy(bufferToEmit, 0, 0, self._currentMessage.getResponseSize());
				self._receivingDataBuffer = self._receivingDataBuffer.slice(self._currentMessage.getResponseSize());
				
				self._currentMessage.setResponse(bufferToEmit);
				self.emit('data', self._currentMessage);
				self._currentMessage = null;
				
			}else{
				self._debug('waiting for more data with command: ' + self._currentMessage.getName());
				self._debug('already received: ' + self._receivingDataBuffer.length);
			}
		})
		.connect(port, ip);
	
};

EthernetConnection.prototype._destroy = function(){
	//reset buffers and state vars
	this._endMessageTimer();
	this._resetMessageBuffer();
	this._resetReceiveBuffer();
	this._currentMessage = null;

	this._sensorSocket.destroy();
};

EthernetConnection.prototype.disconnect = function(){
	this._destroy();
};

EthernetConnection.prototype.isBusy = function(){
	return this._currentMessage !== null;
};


EthernetConnection.prototype.sendMessage = function(message){	
	//not busy or blocking message -> add to buffer
	if (!this._connected)
		return false;

	//can execute only streamStop while receiving stream
	if (this._currentMessage && this._currentMessage.getName() === STREAM_MSG_NAME) {
		if (message.getName() !== STREAM_STOP_MSG_NAME) {
			return false;
		}
	}

	if (!this._currentMessage || message.isBlocking()) {
		this._messageBuffer.push(message);
		this._debug('Adding command to buffer: ' + message.getName() + ', blocking: '  + message.getBlocking());
		this._debug('MeesageBufferLength: ' + this._messageBuffer.length);
	}else{
		this._debug('Not-blocking message: ' + message.getName() + ' omitted cause other command being executed: ' + this._currentMessage.getName());
	}
};

//exposure
module.exports = EthernetConnection;