"use strict";

var assign = require('object-assign');
var util = require('util');
var BaseCanvas = require('./baseCanvas');
var UnitsConverter = require('../helpers/UnitsConverter');
var GlobalConstants = require('../constants/GlobalConstants');



util.inherits(StreamCanvas, BaseCanvas);


function StreamCanvas(domNode) {
	BaseCanvas.call(this, domNode);


	this._streamData = null;
	this._avgMin = null;
	this._avgMax = null;
	this._avgMean = null;

	// this._units = UnitsConverter.MICRONS;
	
	this._pixelsPerMicronValue = 1;

	//omit every Nth frame
	this._frameCounter = 0;

	//value zoom area
	this._viewArea = {
		from : 0,
		to : 64000
	};


	this._init();

};

//sensor range
StreamCanvas.VALS_RANGE = 64000;

//paper js constants
StreamCanvas.CANVAS_COLOR = '#000';
StreamCanvas.VALID_POINT_COLOR = [255, 255, 255, 255];
StreamCanvas.INVALID_POINT_COLOR = [255, 30, 30, 255];
StreamCanvas.AVG_COLOR = '#ffec23';


StreamCanvas.prototype.setStreamData = function(data){
    this._streamData = data;
};

StreamCanvas.prototype.setAvgMin = function (avgMin) {
	this._avgMin = avgMin;
};

StreamCanvas.prototype.setAvgMax = function (avgMax) {
	this._avgMax = avgMax;
};

StreamCanvas.prototype.setAvgMean = function (avgMean) {
	this._avgMean = avgMean;
};

StreamCanvas.prototype.resetData = function () {
    this._streamData = null;
	this._avgMin = null;
	this._avgMax = null;
	this._avgMean = null;
};

StreamCanvas.prototype.refresh = function(w, h){
	//activate scope on actions which are drawing directly and are calle from outside
	this._paperScope.activate();
	this._refresh(w, h);
};

StreamCanvas.prototype.getSvg = function(){
	return this._paperScope.project.exportSVG({asString:true});
};

StreamCanvas.prototype.getPng = function(){
	var imageData = this._canvasDomNode.toDataURL();

	//remove encoding string and decode data
	imageData = imageData.split(',')[1];
	imageData = atob(imageData);

	return imageData;
};



StreamCanvas.prototype._refresh = function(w, h){
	w = (w) ? w : this.VIEW_WIDTH;
	h = (h) ? h : this.VIEW_HEIGHT;

	this._resizeView(w, h);

	//resize background
	this._elements.background.setBounds([0, 0, w, h]);

	this._elements.stream.setSize(new paper.Size(this.VIEW_WIDTH, this.VIEW_HEIGHT));
	this._elements.stream.set({
		position : new paper.Point(this.VIEW_WIDTH / 2, this.VIEW_HEIGHT / 2)
	});

	//update view
	this._paperScope.view.update();
};



StreamCanvas.prototype._init = function(){
	this._elements.background = new paper.Path.Rectangle({
		from : [0, 0],
		to : [this.VIEW_WIDTH, this.VIEW_HEIGHT],
		fillColor : StreamCanvas.CANVAS_COLOR
	});

	this._elements.stream = new paper.Raster(null, [this.VIEW_WIDTH / 2, this.VIEW_HEIGHT / 2]);
	this._elements.stream.setSize(new paper.Size(this.VIEW_WIDTH, this.VIEW_HEIGHT));

	// this._elements.streamValid = new paper.Group();
	// this._elements.streamInvalid = new paper.Group();

	this._elements.avgMin = new paper.Path({
		strokeColor : StreamCanvas.AVG_COLOR
	});
	this._elements.avgMax = new paper.Path({
		strokeColor : StreamCanvas.AVG_COLOR
	});
	this._elements.avgMean = new paper.Path({
		strokeColor : StreamCanvas.AVG_COLOR
	});

	this._paperScope.view.onFrame = this._onFrame.bind(this);	
};


StreamCanvas.prototype._drawStream = function () {
	var context = this._canvasDomNode.getContext('2d');
	var canvasData = context.createImageData(this.VIEW_WIDTH, this.VIEW_HEIGHT);

	var streamData = this._streamData ? this._streamData : [];
	var pixelPerData = this.VIEW_WIDTH / streamData.length;

	var shouldCompare = (
		typeof this._avgMax === 'number' &&
		typeof this._avgMin === 'number' &&
		typeof this._avgMean === 'number'
	);

	var color;

	for(var i = 0, n = streamData.length; i < n; i++) {
		
		var posX = Math.round(i * pixelPerData);
		var posY = this.VIEW_HEIGHT - Math.round(((streamData[i] - this._viewArea.from)  * this._pixelsPerMicronValue));

		if (
			!shouldCompare || 
			(streamData[i] > this._avgMin && streamData[i] < this._avgMax)
		){
			color = StreamCanvas.VALID_POINT_COLOR;
		}else{
			color = StreamCanvas.INVALID_POINT_COLOR;			
		}

		canvasData.data[posY * this.VIEW_WIDTH * 4 + posX * 4] = color[0];
		canvasData.data[posY * this.VIEW_WIDTH * 4 + posX * 4 + 1] = color[1];
		canvasData.data[posY * this.VIEW_WIDTH * 4 + posX * 4 + 2] = color[2];
		canvasData.data[posY * this.VIEW_WIDTH * 4 + posX * 4 + 3] = color[3];
	}

	this._elements.stream.setImageData(canvasData);
};

StreamCanvas.prototype._drawAvgs = function () {
	this._elements.avgMax.removeSegments();
	this._elements.avgMin.removeSegments();
	this._elements.avgMean.removeSegments();

	if (this._avgMax){
		this._elements.avgMax.addSegments([
			[0, this.VIEW_HEIGHT - Math.round((this._avgMax - this._viewArea.from) * this._pixelsPerMicronValue) + 0.5],
			[this.VIEW_WIDTH, this.VIEW_HEIGHT - Math.round((this._avgMax - this._viewArea.from) * this._pixelsPerMicronValue)]
		]);
	}

	if (this._avgMin){
		this._elements.avgMin.addSegments([
			[0, this.VIEW_HEIGHT - Math.round((this._avgMin - this._viewArea.from) * this._pixelsPerMicronValue) + 0.5],
			[this.VIEW_WIDTH, this.VIEW_HEIGHT - Math.round((this._avgMin - this._viewArea.from) * this._pixelsPerMicronValue) + 0.5]
		]);
	}

	if (this._avgMean){
		this._elements.avgMean.addSegments([
			[0, this.VIEW_HEIGHT - Math.round((this._avgMean - this._viewArea.from) * this._pixelsPerMicronValue) + 0.5],
			[this.VIEW_WIDTH, this.VIEW_HEIGHT - Math.round((this._avgMean - this._viewArea.from) * this._pixelsPerMicronValue)  + 0.5]
		]);
	}
};

StreamCanvas.prototype._findValidMin = function (numbers) {
	var sorted = numbers.slice().sort(function(a, b){
        if(a === null || isNaN(a) || a === 0) return 1;
        if(b === null || isNaN(b) || b === 0) return -1;
        return a-b;
    });
    return sorted[0];
};

StreamCanvas.prototype._findValidMax = function (numbers) {
	var sorted = numbers.slice().sort(function(a, b){
        if(a === null || isNaN(a) || a === 64000) return 1;
        if(b === null || isNaN(b) || b === 64000) return -1;
        return b-a;
    });
    return sorted[0];
};

StreamCanvas.prototype._onFrame = function (event) {
	this._frameCounter++;
	if (this._frameCounter % 3 !== 0) {
		return;
	}


	// recalculate viewArea and pixelsPerMicronValue ratio
	if (this._streamData){
		var validMin = this._findValidMin(this._streamData);
		var validMax = this._findValidMax(this._streamData);
		var diff = validMax - validMin;
		
		this._viewArea.from = ((validMin - diff / 20) > 0 ) ? (validMin - diff / 20) : 0;
		this._viewArea.to = ((validMax + diff / 20) < StreamCanvas.VALS_RANGE) ? (validMax + diff / 20) : StreamCanvas.VALS_RANGE;

		this._pixelsPerMicronValue = this.VIEW_HEIGHT / (this._viewArea.to - this._viewArea.from);

	}
	
	//profile and template redraw
	this._drawStream();
	this._drawAvgs();
};



module.exports = StreamCanvas;
