"use strict";

var assign = require('object-assign');
var util = require('util');
var BaseCanvas = require('./baseCanvas');
var UnitsConverter = require('../helpers/UnitsConverter');
var GlobalConstants = require('../constants/GlobalConstants');



util.inherits(ProfileCanvas, BaseCanvas);


function ProfileCanvas(domNode) {
	BaseCanvas.call(this, domNode);


	this._profileRawData = null;
	this._profileNormalizedData = null;
	this._profileNormalizedFilteredData = null;
	this._templateData = null;

	this._threshold = null;
	this._activeLines = {from:20, to:100};

	this._displayRawProfile = false;
	this._displayNormalizedProfile = false;
	this._displayNormalizedFilteredProfile = true;

	// this._theoreticalFps = 0;
	this._fps = 0;
	this._units = UnitsConverter.MICRONS;

	this._pixelsPerMicronWidth = 1;
	this._pixelsPerMicronHeight = 1;

	//omit every Nth frame
	this._frameCounter = 0;

	//zoom state
	this._zoomState = ProfileCanvas.ZOOM_CANCELLED;

	//zoom area
	this._viewArea = {
		x : 0,
		y : 0,
		width : ProfileCanvas.DISTANCE_RANGE,
		height : ProfileCanvas.HEIGHT_RANGE
	}; 

	this._init();

};
//profile types range
ProfileCanvas.PROFILE_RAW = 'profileRaw';
ProfileCanvas.PROFILE_NORMALIZED = 'profileNormalized';
ProfileCanvas.PROFILE_NORMALIZED_FILTERED = 'profileNormalizedFiltered';
ProfileCanvas.PROFILE_TEMPLATE = 'template';

//sensor range
ProfileCanvas.DISTANCE_RANGE = 2092;
ProfileCanvas.HEIGHT_RANGE = 2560;

//paper js constants
ProfileCanvas.CANVAS_COLOR = '#000';
ProfileCanvas.GRID_COLOR = '#111111';
ProfileCanvas.GRID_TEXT_COLOR = '#999999';
ProfileCanvas.GRID_NUM_CELLS_DISTANCE = 4;
ProfileCanvas.GRID_NUM_CELLS_HEIGHT = 4;
ProfileCanvas.ZOOM_RECTANGLE_COLOR = '#555555';

ProfileCanvas.TR_LINE_COLOR = '#990000';
ProfileCanvas.TR_LINE_COLOR_HOVER = '#ff0000';

//zoom phase
ProfileCanvas.ZOOM_CANCELLED = 'zoomCancelled';
ProfileCanvas.ZOOM_SELECTION = 'zoomSelection';
ProfileCanvas.ZOOM_VIEW = 'zoomView';

//element names
ProfileCanvas.ELEMENT_THRESHOLD = 'threshold'

//event names
ProfileCanvas.EVENT_TRESHOLD_CHANGE = 'trChange';

ProfileCanvas.prototype.setProfileRaw = function(profile){
    this._profileRawData = profile;
};

ProfileCanvas.prototype.setProfileNormalized = function(profile){
    this._profileNormalizedData = profile;
};

ProfileCanvas.prototype.setProfileNormalizedFiltered = function(profile){
    this._profileNormalizedFilteredData = profile;
};

ProfileCanvas.prototype.setThreshold = function(threshold){
    this._threshold = threshold;
};
ProfileCanvas.prototype.setActiveLines = function(activeLines){
    this._activeLines = activeLines;
};

ProfileCanvas.prototype.setTemplate = function(template){
    this._templateData = template;
};

ProfileCanvas.prototype.cancelTemplate = function(){
    this._templateData = null;
};

ProfileCanvas.prototype.showRawProfile = function(){
    this._displayRawProfile = true;
};

ProfileCanvas.prototype.hideRawProfile = function(){
    this._displayRawProfile = false;
};

ProfileCanvas.prototype.showNormalizedProfile = function(){
    this._displayNormalizedProfile = true;
};

ProfileCanvas.prototype.hideNormalizedProfile = function(){
    this._displayNormalizedProfile = false;
};

ProfileCanvas.prototype.showNormalizedFilteredProfile = function(){
    this._displayNormalizedFilteredProfile = true;
};

ProfileCanvas.prototype.hideNormalizedFilteredProfile = function(){
    this._displayNormalizedFilteredProfile = false;
};

// ProfileCanvas.prototype.setTheoreticalFps = function(fps){
//     this._theoreticalFps = fps;
// };

ProfileCanvas.prototype.setFps = function(fps){
    this._fps = fps;
};

ProfileCanvas.prototype.refresh = function(w, h){
	//activate scope on actions which are drawing directly and are calle from outside
	this._paperScope.activate();
	this._refresh(w, h);
};

ProfileCanvas.prototype.getSvg = function(){
	return this._paperScope.project.exportSVG({asString:true});
};

ProfileCanvas.prototype.getPng = function(){
	var imageData = this._canvasDomNode.toDataURL();

	//remove encoding string and decode data
	imageData = imageData.split(',')[1];
	imageData = atob(imageData);

	return imageData;
};


ProfileCanvas.prototype.setZoom = function(){
	//set only if not already in zoom view
   	if (this._zoomState === ProfileCanvas.ZOOM_CANCELLED) {
   		this._zoomState = ProfileCanvas.ZOOM_SELECTION;
   	}
};

ProfileCanvas.prototype.zoomPlus = function(){
	var widthDiff = this._viewArea.width / 10;
	var heightDiff = this._viewArea.height / 10;

	this._viewArea.x +=  widthDiff / 2;
	this._viewArea.width -=  widthDiff;

	this._viewArea.y +=  heightDiff / 2;
	this._viewArea.height -=  heightDiff;
	
	this._zoomState = ProfileCanvas.ZOOM_VIEW;
	this._refresh();
};

ProfileCanvas.prototype.zoomMinus = function(){
	var widthDiff = this._viewArea.width / 9;
	var heightDiff = this._viewArea.height / 9;
	
	this._viewArea.x -=  widthDiff / 2;
	this._viewArea.width +=  widthDiff;

	this._viewArea.y -=  heightDiff / 2;
	this._viewArea.height +=  heightDiff;

	if (this._viewArea.x + this._viewArea.width > ProfileCanvas.DISTANCE_RANGE) {
		this._viewArea.width = ProfileCanvas.DISTANCE_RANGE - this._viewArea.x;
	}
	if (this._viewArea.y + this._viewArea.height > ProfileCanvas.HEIGHT_RANGE) {
		this._viewArea.height = ProfileCanvas.HEIGHT_RANGE - this._viewArea.y;
	}
	if (this._viewArea.x < 0) {
		this._viewArea.width += this._viewArea.x;
		this._viewArea.x = 0;
	}
	if (this._viewArea.y < 0) {
		this._viewArea.height += this._viewArea.y;
		this._viewArea.y = 0;
	}

	this._zoomState = ProfileCanvas.ZOOM_VIEW;
	this._refresh();
};

ProfileCanvas.prototype.cancelZoom = function(){
	this._zoomState = ProfileCanvas.ZOOM_CANCELLED;
	this._viewArea = {
		x : 0,
		y : 0,
		width : ProfileCanvas.DISTANCE_RANGE,
		height : ProfileCanvas.HEIGHT_RANGE
	};
	this.refresh();
};

ProfileCanvas.prototype._formatFpsText = function(){
	return /*this._theoreticalFps + " fps\n" + */this._fps +  " fps";
};

ProfileCanvas.prototype._emitTresholdChange = function(mouseY){
	this.emit(
		ProfileCanvas.EVENT_TR_LINE_CHANGE,
		this._getRealThreshold(mouseY)
	);
};

ProfileCanvas.prototype._getRealThreshold = function(mouseY){
	//ratio of currently vieved area and target canvas size
	var viewAreaRatio = this._viewArea.height / this.TARGET_HEIGHT;

	//reverse number from this.HEIGHT_RANGE becaseu threshold 0 is down
	return Math.round(ProfileCanvas.HEIGHT_RANGE - (this._viewArea.y + this._toTargetSize(mouseY) * viewAreaRatio));
};

ProfileCanvas.prototype._refresh = function(w, h){
	w = (w) ? w : this.VIEW_WIDTH;
	h = (h) ? h : this.VIEW_HEIGHT;

	this._resizeView(w, h);
	this._pixelsPerMicronWidth = w / ((this._viewArea) ? this._viewArea.width : ProfileCanvas.DISTANCE_RANGE);
	this._pixelsPerMicronHeight = h / ((this._viewArea) ? this._viewArea.height : ProfileCanvas.HEIGHT_RANGE);

	console.log('resize', this._pixelsPerMicronHeight, this._pixelsPerMicronWidth);

	//resize background
	this._elements.background.setBounds([0, 0, w, h]);

	//resize fps
	// this._elements.fpsText.setPosition(new paper.Point([w - 85, 30]));
	this._elements.fpsValues.setPosition(new paper.Point([w - 40, 30]));

	//redraw grid
	this._elements.grid.removeChildren();
	this._elements.grid.remove();
	this._elements.grid = this._drawGrid(
		((this._viewArea) ? this._viewArea.width : ProfileCanvas.DISTANCE_RANGE),
		((this._viewArea) ? this._viewArea.height : ProfileCanvas.HEIGHT_RANGE),
		((this._viewArea) ? this._viewArea.x : null),
		((this._viewArea) ? this._viewArea.y : null)
	);

	//profiles to fron
	this._elements.template.bringToFront();
	this._elements.profileRaw.bringToFront();
	this._elements.profileNormalized.bringToFront();
	this._elements.profileNormalizedFiltered.bringToFront();
	
	//update view
	this._paperScope.view.update();
};



ProfileCanvas.prototype._init = function(){
	this._elements.background = new paper.Path.Rectangle({
		from : [0, 0],
		to : [this.VIEW_WIDTH, this.VIEW_HEIGHT],
		fillColor : ProfileCanvas.CANVAS_COLOR
	});

	// this._elements.fpsText = new paper.PointText({
	// 	point : [this.VIEW_WIDTH - 85, 30],
	// 	content : "Sensor:", //"Sensor:\nStudio:",
	// 	fillColor : ProfileCanvas.GRID_TEXT_COLOR,
	// 	justification : 'right'
	// });

	this._elements.fpsValues = new paper.PointText({
		point : [this.VIEW_WIDTH - 40, 30],
		content : this._formatFpsText(),
		fillColor : ProfileCanvas.GRID_TEXT_COLOR,
		justification : 'right'
	});

	this._elements.grid = this._drawGrid(ProfileCanvas.DISTANCE_RANGE, ProfileCanvas.HEIGHT_RANGE);

	this._elements.threshold = new paper.Path({
		strokeColor: ProfileCanvas.TR_LINE_COLOR,
		name : ProfileCanvas.ELEMENT_THRESHOLD
	});
	this._elements.template = new paper.Path({
		strokeColor: '#21BA45'
	});
	this._elements.profileRaw = new paper.Path({
		strokeColor: '#DB2828'
	});
	this._elements.profileNormalized = new paper.Path({
		strokeColor: '#555555'
	});
	this._elements.profileNormalizedFiltered = new paper.Path({
		strokeColor: '#2185D0'
	});

	this._paperScope.view.onFrame = this._onFrame.bind(this);

	var item, segment, hitTool, hitResult, zoomDrag = false;
	var tmpZoomSelection;
	hitTool = new paper.Tool();
	hitTool.activate();

	hitTool.onMouseMove = function(event){
		var hitResult = paper.project.hitTest(
				event.point, {
					segments: false,
					stroke: true,
					fill: false,
					tolerance: 5
				}
			);

		//THRESHOLD change OFF
		//if hit with threshold line, set hover color
		// if (hitResult && hitResult.item.name === ProfileCanvas.ELEMENT_THRESHOLD) {
		// 	this._elements.threshold.strokeColor = ProfileCanvas.TR_LINE_COLOR_HOVER;
		// }else{
		// 	this._elements.threshold.strokeColor = ProfileCanvas.TR_LINE_COLOR;
		// }
	}.bind(this);

	hitTool.onMouseDown = function(event) {
		item = segment = null;

		//if dragging in any zoom state
		if (event.event.button === 2 && (this._zoomState === ProfileCanvas.ZOOM_VIEW || this._zoomState === ProfileCanvas.ZOOM_SELECTION)) {
			zoomDrag = true;
			document.body.style.cursor = "all-scroll";
			return;
		}

		//else detect hit with any item
		var hitResult = paper.project.hitTest(
			event.point,
			{
				segments: false,
				stroke: true,
				fill: true,
				tolerance: 5
			}
		);

		if (!hitResult) {
			return;
		}else {
			//hit with object found
			//reference to hit item
			item = hitResult.item;
			//was it segment of area?
			if (hitResult.type == 'segment' && item.selected) {
				segment = hitResult.segment;
			}
		}

	}.bind(this);

	hitTool.onMouseDrag = function(event) {

		//drag while zoom dragging - do nothing
		if (zoomDrag) {
			return;
		}

		//else move rectangle zoom selection
		if (this._zoomState === ProfileCanvas.ZOOM_SELECTION) {
			if (tmpZoomSelection){
				tmpZoomSelection.remove();
				tmpZoomSelection = null;
			}
			tmpZoomSelection = new paper.Path.Rectangle({
				from : event.downPoint,
				to : event.point,
				strokeColor : ProfileCanvas.ZOOM_RECTANGLE_COLOR
			});
		}else if (item) {
			switch (item.name) {
				//THRESHOLD change OFF
				// case ProfileCanvas.ELEMENT_THRESHOLD:
					// // if moved cursor - update cursor position value and update intersections
					// if (event.point.y > 0 && event.point.y < this.VIEW_HEIGHT) {
					// 	this._emitTresholdChange(event.point.y);
					// }
					// break;
			}
		}
	}.bind(this);


	hitTool.onMouseUp = function(event) {
		if (zoomDrag) {
			zoomDrag = false;
			document.body.style.cursor = "default";

			//calculate x and y offsets - Y is reverted cause zero is on bottom
			var offsetX = (event.downPoint.x - event.point.x) / this._pixelsPerMicronWidth;
			var offsetY = -(event.downPoint.y - event.point.y) / this._pixelsPerMicronHeight;

			this._viewArea.x +=  offsetX;
			this._viewArea.y +=  offsetY;

			if (this._viewArea.x + this._viewArea.width > ProfileCanvas.DISTANCE_RANGE) {
				this._viewArea.x = ProfileCanvas.DISTANCE_RANGE - this._viewArea.width;
			}
			if (this._viewArea.y + this._viewArea.height > ProfileCanvas.HEIGHT_RANGE) {
				this._viewArea.y = ProfileCanvas.HEIGHT_RANGE - this._viewArea.height;
			}
			if (this._viewArea.x < 0) {
				this._viewArea.x = 0;
			}
			if (this._viewArea.y < 0) {
				this._viewArea.y = 0;
			}

			this._refresh();

		}else if (tmpZoomSelection) {
			this._viewArea = this._computeZoomArea(tmpZoomSelection);
			this._zoomState = ProfileCanvas.ZOOM_VIEW;
			tmpZoomSelection.remove();
			tmpZoomSelection = null;

			this._refresh();
		}
	}.bind(this);
};

ProfileCanvas.prototype._computeZoomArea = function(zoomSelectionRectange){
	var area = {
		x : zoomSelectionRectange.bounds.x / this._pixelsPerMicronWidth,
		y : (this.VIEW_HEIGHT - zoomSelectionRectange.bounds.y - zoomSelectionRectange.bounds.height) / this._pixelsPerMicronHeight,
		width : zoomSelectionRectange.bounds.width / this._pixelsPerMicronWidth,
		height : zoomSelectionRectange.bounds.height / this._pixelsPerMicronHeight
	};

	// area.x = Math.floor(area.x / 100) * 100;
	// area.y = Math.floor(area.y / 100) * 100;
	// area.width = Math.ceil(area.width / 100) * 100;
	// area.height = Math.ceil(area.height / 100) * 50;

	if (area.x + area.width > ProfileCanvas.DISTANCE_RANGE) {
		area.width = ProfileCanvas.DISTANCE_RANGE - area.x;
	}
	if (area.y + area.height > ProfileCanvas.HEIGHT_RANGE) {
		area.height = ProfileCanvas.HEIGHT_RANGE - area.y;
	}
	if (area.x < 0) {
		area.width += area.x;
		area.x = 0;
	}
	if (area.y < 0) {
		area.height += area.y;
		area.y = 0;
	}
	
	return area;
};

ProfileCanvas.prototype._drawGrid = function(distanceRange, heightRange, distanceOffset, heightOffset) {
		var gridGroup = new paper.Group();

		//offsets are optional
		distanceOffset = distanceOffset ? distanceOffset : 0;
		heightOffset = heightOffset ? heightOffset : 0;

		var gridCellDistance = (distanceRange / ProfileCanvas.GRID_NUM_CELLS_DISTANCE);
		var gridCellHeight = (heightRange / ProfileCanvas.GRID_NUM_CELLS_HEIGHT);

		//helper function to format grid value
		var formatGridValue = function (value) {
			return UnitsConverter.toDisplay(this._units, value, true);
		}.bind(this);

		var columnCoord, rowCoord;

		//grid matrix
		for (var i = 1; i < ProfileCanvas.GRID_NUM_CELLS_DISTANCE; i++) {
			columnCoord = Math.round((this.VIEW_WIDTH / ProfileCanvas.GRID_NUM_CELLS_DISTANCE) * i) + 0.5;
			gridGroup.addChild(new paper.Path({
				segments : [
						[columnCoord, 0],
						[columnCoord, this.VIEW_HEIGHT]
					],
				strokeColor : ProfileCanvas.GRID_COLOR
			}));

			//horizontal values on grid
			// gridGroup.addChild(new paper.PointText({
			// 	point : [columnCoord + 5, this.VIEW_HEIGHT - 10],
			// 	content : formatGridValue(distanceOffset + i * gridCellDistance),
			// 	fillColor : ProfileCanvas.GRID_TEXT_COLOR,
			// 	justification : 'left'
			// }));
		}

		for (var i = 1; i < ProfileCanvas.GRID_NUM_CELLS_HEIGHT; i++) {
			rowCoord = Math.round((this.VIEW_HEIGHT / ProfileCanvas.GRID_NUM_CELLS_HEIGHT) * i) + 0.5;

			gridGroup.addChild(new paper.Path({
				segments : [
						[0, rowCoord],
						[this.VIEW_WIDTH, rowCoord]
					],
				strokeColor : ProfileCanvas.GRID_COLOR
			}));

			if (i > 1) {
				// gridGroup.addChild(new paper.PointText({
				// 	point : [5, this.VIEW_HEIGHT - rowCoord - 10],
				// 	content : formatGridValue(heightOffset + i * gridCellHeight),
				// 	fillColor : ProfileCanvas.GRID_TEXT_COLOR,
				// 	justification : 'left'			
				// }));
			}
		}

		//axises lines in corner		
		gridGroup.addChild(new paper.Path({
			segments : [
				[10.5, this.VIEW_HEIGHT - 30],
				[10.5, this.VIEW_HEIGHT - 10.5],
				[30, this.VIEW_HEIGHT - 10.5]
			],
			strokeColor : ProfileCanvas.GRID_TEXT_COLOR
		}));

		//axises labels
		gridGroup.addChild(new paper.PointText({
			point : [35, this.VIEW_HEIGHT - 7.5],
			content : 'PIX',
			fillColor : ProfileCanvas.GRID_TEXT_COLOR,
			justification : 'left'
		}));
		gridGroup.addChild(new paper.PointText({
			point : [15, this.VIEW_HEIGHT - 35],
			rotation : -90,
			justification : 'left',
			content : 'LIGHT',
			fillColor : ProfileCanvas.GRID_TEXT_COLOR
		}));

		return gridGroup;
	};

ProfileCanvas.prototype._drawProfile = function (type) {

	
	switch (type){
		case ProfileCanvas.PROFILE_RAW:
			var profileData = this._profileRawData;
			var element = 'profileRaw';
			break;

		case ProfileCanvas.PROFILE_NORMALIZED_FILTERED:
			var profileData = this._profileNormalizedFilteredData;
			var element = 'profileNormalizedFiltered';
			break;

		case ProfileCanvas.PROFILE_NORMALIZED:
			var profileData = this._profileNormalizedData;
			var element = 'profileNormalized';
			break;

		case ProfileCanvas.PROFILE_TEMPLATE:
			var profileData = this._templateData;
			var element = 'template';
			break;
	}

	//prepare data and empty segments of selected paperjs element
	var profileData = profileData ? profileData : [];
	this._elements[element].removeSegments();

	//don't draw unfiltered if not set
	if (type == ProfileCanvas.PROFILE_RAW && !this._displayRawProfile){
		return;
	}

	if (type == ProfileCanvas.PROFILE_NORMALIZED && !this._displayNormalizedProfile){
		return;
	}

	if (type == ProfileCanvas.PROFILE_NORMALIZED_FILTERED && !this._displayNormalizedFilteredProfile){
		return;
	}

	for(var i = 0, n = profileData.length; i < n; i++) {
		//show everything even inactive part of line to see what is going on
		// if (
		// 	i > this._activeLines.from &&
		// 	i < this._activeLines.to 
			//we need to draw all points even when they are out of zoom area, cause otherwise the line withut some segments is shitty with holes
			// i > this._viewArea.x &&
			// i < (this._viewArea.x + this._viewArea.width) &&
			// profileData[i] >= this._viewArea.y &&
			// profileData[i] <= (this._viewArea.y + this._viewArea.height)
		// ) {
			var rowProfileX = Math.round((i - this._viewArea.x) * this._pixelsPerMicronWidth);
			var rowProfileY = this.VIEW_HEIGHT - Math.round((profileData[i] - this._viewArea.y) * this._pixelsPerMicronHeight);

			this._elements[element].addSegment([rowProfileX, rowProfileY]);
		// }
	}

};

ProfileCanvas.prototype._drawThreshold = function () {
	this._elements.threshold.removeSegments();
	this._elements.threshold.addSegments([
		[0, this.VIEW_HEIGHT - Math.round((this._threshold - this._viewArea.y) * this._pixelsPerMicronHeight) + 0.5],
		[this.VIEW_WIDTH, this.VIEW_HEIGHT - Math.round((this._threshold - this._viewArea.y) * this._pixelsPerMicronHeight)  + 0.5]
	]);
	this._elements.threshold.bringToFront();
};



ProfileCanvas.prototype._onFrame = function (event) {
	this._frameCounter++;
	if (this._frameCounter % 3 !== 0) {
		return;
	}
	
	this._drawThreshold();

	//profile and template redraw
	this._drawProfile(ProfileCanvas.PROFILE_RAW);
	//this._drawProfile(ProfileCanvas.PROFILE_NORMALIZED);
	this._drawProfile(ProfileCanvas.PROFILE_NORMALIZED_FILTERED);
	this._drawProfile(ProfileCanvas.PROFILE_TEMPLATE);


	//fps redraw
	this._elements.fpsValues.content = this._formatFpsText();
};



module.exports = ProfileCanvas;
