"use strict";

var assign = require('object-assign');
var util = require('util');
var BaseCanvas = require('./baseCanvas');
var UnitsConverter = require('../helpers/UnitsConverter');
var GlobalConstants = require('../constants/GlobalConstants');


util.inherits(HistoryCanvas, BaseCanvas);


function HistoryCanvas(domNode, appConfig) {
	BaseCanvas.call(this, domNode);

	/**
	 * Timestamp when pause was set for redraw in pause
	 * @type {Number}
	 */
	this._pausedNow = null;
	this._pause = false;

	/**
	 * Points params
	 */

	this._data = [];
	this._minDataDistance = [];

	/**
	 * App config for formatting using correct units
	 */
	this._appConfig = appConfig;

	/**
	 * Displayed timespan in milliseconds
	 * @type {Number}
	 */
	this._timeSpan = 5000;

	//zoom state
	this._zoomState = HistoryCanvas.ZOOM_CANCELLED;

	//zoom area - on horizontal axis
	this._viewArea = {
		from : 0,
		to : Number.MAX_SAFE_INTEGER
	};

	//disable
	this._disableValuesAutoZoom = false;

	this._cursorPosition = 20;
	this._intersection = null;

	//omit every Nth frame
	this._frameCounter = 0;

	this._init();

};

//sensor max distance/height range
HistoryCanvas.VALS_RANGE = 30000;

//paper js constants
HistoryCanvas.CANVAS_COLOR = '#F0F0F0';
HistoryCanvas.GRID_COLOR = '#DDD';
HistoryCanvas.GRID_TEXT_COLOR = '#BBB';
HistoryCanvas.GRID_NUM_CELLS_VALS = 10;
HistoryCanvas.GRID_NUM_CELLS_TIME = 10;
HistoryCanvas.CURSOR_COLOR = '#F2711C';
HistoryCanvas.CURSOR_SIZE = 18;
HistoryCanvas.ZOOM_RECTANGLE_COLOR = '#555555';
HistoryCanvas.POINTS_CIRCLE_RADIUS = 2.5;

//zoom phase
HistoryCanvas.ZOOM_CANCELLED = 'zoomCancelled';
HistoryCanvas.ZOOM_SELECTION = 'zoomSelection';
HistoryCanvas.ZOOM_VIEW = 'zoomView';

//paper js named elements
HistoryCanvas.ELEMENT_CURSOR = 'elementCursor';

//events
HistoryCanvas.EVENT_CURSOR_POINTS = 'eventCursorPoints';

//other
HistoryCanvas.CURSOR_POSITION_TEXT = 'Cursor position: ';

HistoryCanvas.prototype.setTimeSpan = function(timeSpan){
    this._timeSpan = timeSpan * 1000;

    //redraw grid
	this._elements.grid.removeChildren();
	this._elements.grid.remove();
	this._elements.grid = this._drawGrid(this._getValuesZoomRange(this._data));
};

HistoryCanvas.prototype.setMaxRange = function(maxRange){
    HistoryCanvas.VALS_RANGE = maxRange;
};

HistoryCanvas.prototype.resetData = function(){
    this._data = [];
};

HistoryCanvas.prototype.setPause = function(){
    this._pause = {
    	now : Date.now(),
    	data : this._data.slice()
    };
};

HistoryCanvas.prototype.resetPause = function(){
    this._pause = null;
};

HistoryCanvas.prototype.addData = function(data){
    this._data.unshift({
    	timestamp : Date.now(),
    	data : data
    });
    // this._updateCursorIntersections();
};


HistoryCanvas.prototype.enableValuesAutoZoom = function () {
	this._disableValuesAutoZoom = false;	
};

HistoryCanvas.prototype.disableValuesAutoZoom = function () {
	this._disableValuesAutoZoom = true;
};

HistoryCanvas.prototype.refresh = function(w, h){
	//activate scope on actions which are drawing directly and are calle from outside
	this._paperScope.activate();
	// this._updateCursorIntersections();
	this._refresh(w, h);
};

HistoryCanvas.prototype.getSvg = function(){
	return this._paperScope.project.exportSVG({asString:true});
};

HistoryCanvas.prototype.getPng = function(){
	var imageData = this._canvasDomNode.toDataURL();

	//remove encoding string and decode data
	imageData = imageData.split(',')[1];
	imageData = atob(imageData);

	return imageData;
};


HistoryCanvas.prototype.setZoom = function(){
	//set only if not already in zoom view
   	if (this._zoomState === HistoryCanvas.ZOOM_CANCELLED) {
   		this._zoomState = HistoryCanvas.ZOOM_SELECTION;
   	}
};

HistoryCanvas.prototype.zoomPlus = function(){
	var currentViewWidth = (this._viewArea.to === Number.MAX_SAFE_INTEGER) ? this._data.length : this._viewArea.to - this._viewArea.from;
	
	if (currentViewWidth < 15){
		return;
	}
	var change = Math.round(currentViewWidth / 15);

	this._viewArea.from += change;
	this._viewArea.to = ((this._viewArea.to === Number.MAX_SAFE_INTEGER) ? this._data.length : this._viewArea.to) - change;
	
	// console.log(this._viewArea);

	this._zoomState = HistoryCanvas.ZOOM_VIEW;
	// this._updateCursorIntersections();
	this._refresh();
};

HistoryCanvas.prototype.zoomMinus = function(){
	var currentViewWidth = (this._viewArea.to === Number.MAX_SAFE_INTEGER) ? this._data.length : this._viewArea.to - this._viewArea.from;
	
	var change = Math.max(Math.round(currentViewWidth / 20), 1);

	this._viewArea.from -= change;
	this._viewArea.to = ((this._viewArea.to === Number.MAX_SAFE_INTEGER) ? this._data.length : this._viewArea.to) + change;

	if (this._viewArea.to >= this._data.length) {
		this._viewArea.to = this._data.length;
	}
	if (this._viewArea.from < 0) {
		this._viewArea.from = 0;
	}
	
	this._zoomState = HistoryCanvas.ZOOM_VIEW;
	// this._updateCursorIntersections();
	this._refresh();
};

HistoryCanvas.prototype.cancelZoom = function(){
	this._zoomState = HistoryCanvas.ZOOM_CANCELLED;
	this._viewArea = {
		from : 0,
		to : Number.MAX_SAFE_INTEGER
	};
	// this._updateCursorIntersections();
	this.refresh();
};

// HistoryCanvas.prototype._getSampleRateText = function(){
// 	var sampleRate = 0;

// 	if (this._data.length && !this._pause){
// 		var now = Date.now();
// 		this._data.forEach(function (point) {
// 			if (now - point.timestamp < 1000){
// 				sampleRate++;
// 			} 
// 		});
// 	}

// 	return sampleRate + ' fps';

// };


HistoryCanvas.prototype._refresh = function(w, h){
	w = (w) ? w : this.VIEW_WIDTH;
	h = (h) ? h : this.VIEW_HEIGHT;

	this._resizeView(w, h);

	//resize background
	this._elements.background.setBounds([0, 0, w, h]);

	//cursor position text
	this._elements.cursorPositionText.setPosition(new paper.Point([w - 75, 25]));

	//update view
	this._paperScope.view.update();
};



HistoryCanvas.prototype._init = function(){

	this._elements.background = new paper.Path.Rectangle({
		from : [0, 0],
		to : [this.VIEW_WIDTH, this.VIEW_HEIGHT],
		fillColor : HistoryCanvas.CANVAS_COLOR
	});

	// this._elements.numberOfSamplesText = new paper.PointText({
	// 	point : [this.VIEW_WIDTH - 105, 30],
	// 	content : "Sample rate:",
	// 	fillColor : HistoryCanvas.GRID_TEXT_COLOR,
	// 	justification : 'right'
	// });

	// this._elements.numberOfSamplesValue = new paper.PointText({
	// 	point : [this.VIEW_WIDTH - 40, 30],
	// 	content : this._getSampleRateText(),
	// 	fillColor : HistoryCanvas.GRID_TEXT_COLOR,
	// 	justification : 'right'
	// });
	
	var valsRange = this._getValuesZoomRange([]);
	

	this._elements.grid = this._drawGrid(valsRange);

	this._elements.zoomRectangleSelection = null;

	this._elements.cursor = new paper.Group();
	this._redrawCursor(valsRange);	


	this._elements.cursorPositionText = new paper.PointText({
		point : [this.VIEW_WIDTH - 75, 25],
		content : HistoryCanvas.CURSOR_POSITION_TEXT + '-',
		fillColor : HistoryCanvas.GRID_TEXT_COLOR,
		justification : 'right'
	});

	this._elements.pointsImage = new paper.Raster(null, [this.VIEW_WIDTH / 2, this.VIEW_HEIGHT / 2]);
	this._elements.points = this._drawPoints();

	this._paperScope.view.onFrame = this._onFrame.bind(this);

	var item, segment, hitTool, hitResult, zoomDrag = false;
	hitTool = new paper.Tool()
	hitTool.activate();

	hitTool.onMouseDown = function(event) {
		item = segment = null;

		//if dragging in any zoom state
		if (event.event.button === 2 && (this._zoomState === HistoryCanvas.ZOOM_VIEW || this._zoomState === HistoryCanvas.ZOOM_SELECTION)) {
			zoomDrag = true;
			document.body.style.cursor = "all-scroll";
			return;
		}

		//else detect hit with any item
		var hitResult = paper.project.hitTest(
			event.point,
			{
				segments: false,
				stroke: false,
				fill: true,
				tolerance: 5
			}
		);

		if (!hitResult) {
			return;
		}else {
			//hit with object found
			//reference to hit item
			item = hitResult.item;
			//was it segment of area?
			if (hitResult.type == 'segment' && item.selected) {
				segment = hitResult.segment;
			}
		}

	}.bind(this);

	hitTool.onMouseDrag = function(event) {
		//drag while zoom dragging - do nothing
		if (zoomDrag) {
			return;
		}

		//else move rectangle zoom selection or move cursor
		if (this._zoomState === HistoryCanvas.ZOOM_SELECTION) {
			if (this._elements.zoomRectangleSelection){
				this._elements.zoomRectangleSelection.remove();
				this._elements.zoomRectangleSelection = null;
			}
			this._elements.zoomRectangleSelection = new paper.Path.Rectangle({
				from : [event.downPoint.x, 0],
				to : [event.point.x, this.VIEW_HEIGHT],
				fillColor : HistoryCanvas.ZOOM_RECTANGLE_COLOR,
				opacity : 0.3 
			});
			this._elements.zoomRectangleSelection.insertAbove(this._elements.grid);
		}else if (item && item.name == HistoryCanvas.ELEMENT_CURSOR && event.point.x > 0 && event.point.x < this.VIEW_WIDTH) {
			// if moved cursor - update cursor position value and update intersections
			this._cursorPosition = Math.round(event.point.x);
			// this._updateCursorIntersections();
		}
	}.bind(this);


	hitTool.onMouseUp = function(event) {
		if (zoomDrag) {
			zoomDrag = false;
			document.body.style.cursor = "default";

			//calculate x and y offsets - Y is reverted cause zero is on bottom
			var currentViewWidth = this._getCurrentViewWidth();
			var currentPixelsPerSample = this._getCurrentPixelsPerSample();
			var offsetX = Math.round((event.downPoint.x - event.point.x) / currentPixelsPerSample);


			this._viewArea.from +=  offsetX;
			this._viewArea.to +=  offsetX;

			if (this._viewArea.to > this._data.length) {
				this._viewArea.to = this._data.length - 1;
				this._viewArea.from = this._viewArea.to - currentViewWidth;
			}
			if (this._viewArea.from < 0) {
				this._viewArea.from = 0;
				this._viewArea.to = currentViewWidth;
			}

			// this._updateCursorIntersections();
			this._refresh();


		}else if (this._elements.zoomRectangleSelection) {
			this._viewArea = this._computeZoomArea(this._elements.zoomRectangleSelection);
			this._zoomState = HistoryCanvas.ZOOM_VIEW;
			this._elements.zoomRectangleSelection.remove();
			this._elements.zoomRectangleSelection = null;

			// this._updateCursorIntersections();
			this._refresh();
		}
	}.bind(this);
};

HistoryCanvas.prototype._computeZoomArea = function(zoomSelectionRectange){
	var currentViewWidth = this._getCurrentViewWidth();
	var currentPixelsPerSample = this._getCurrentPixelsPerSample();
	var area = {
		from : Math.round(
			zoomSelectionRectange.bounds.x / currentPixelsPerSample
		),
		to : Math.round(
			(zoomSelectionRectange.bounds.x + zoomSelectionRectange.bounds.width) / currentPixelsPerSample
		),
	};

	// area.x = Math.floor(area.x / 100) * 100;
	// area.y = Math.floor(area.y / 100) * 100;
	// area.width = Math.ceil(area.width / 100) * 100;
	// area.height = Math.ceil(area.height / 100) * 100;
	var currentViewWidth = this._viewArea.to - this._viewArea.from;
	if (area.to >= this._data.length) {
		area.to = this._data.length;
	}
	if (area.from < 0) {
		area.from = 0;
	}

	return area;
};

HistoryCanvas.prototype._getCurrentViewWidth = function(){
	return this._data.length;
};

HistoryCanvas.prototype._getCurrentPixelsPerSample = function(){
	return this.VIEW_WIDTH / Math.max(this._getCurrentViewWidth() - 1, 1);
};

HistoryCanvas.prototype._getValuesZoomRange = function(data){
	//return full range if no points or values zoom disabled or no axis visible
	if (	
			!data.length ||
			this._disableValuesAutoZoom
		){
		return {
			from : 0,
			to : HistoryCanvas.VALS_RANGE
		}
	}

	//initial values for algoritm
	var tmpFrom = HistoryCanvas.VALS_RANGE,
	    tmpTo = 0;

	//find real min/max in visible points sets
	var oneData = null;
	for (var i = 0; i < data.length; i++) {
		var oneData = data[i];
		if (oneData.data < tmpFrom){
			tmpFrom = oneData.data;
		}
		if (oneData.data > tmpTo){
			tmpTo = oneData.data;
		}
	}


	//ensure space from bottom and top sides of graph
	var fromToDiff = Math.max(Math.round((tmpTo - tmpFrom) / 12), 20); 

	var expectedFrom = tmpFrom - fromToDiff;
	var expectedTo = tmpTo + fromToDiff;

	//round to nice values
	expectedFrom = expectedFrom - expectedFrom % 100; 
	expectedTo = expectedTo + (100 - (expectedTo % 100));

	//return but wit limits 0 and max vals range
	return {
		from : Math.max(0, expectedFrom), 
		to : Math.min(HistoryCanvas.VALS_RANGE, expectedTo)
	}
};

HistoryCanvas.prototype._updateCursorIntersections = function(data, now){

	if (!data || !data.length){
		return;
	}

	var pixelPerMillisecond = this.VIEW_WIDTH / this._timeSpan;

	var minDistance = Number.MAX_SAFE_INTEGER;
	this._intersection = null;
	data.forEach((sample, index) => {
		if (Math.abs(((now - sample.timestamp) * pixelPerMillisecond) - this._cursorPosition) < minDistance) {
			minDistance = Math.abs(((now - sample.timestamp) * pixelPerMillisecond) - this._cursorPosition);
			this._intersection = JSON.parse(JSON.stringify(sample));
		}
	});

	this._intersection.timestamp = now - this._intersection.timestamp; 


	//min distance in pixels
	//reset if too large
	if (minDistance > 30){
		this._intersection = null;
	}

	this.emit(HistoryCanvas.EVENT_CURSOR_POINTS, this._intersection);
};

HistoryCanvas.prototype._drawGrid = function(valsRange) {
	var gridGroup = new paper.Group();
	gridGroup.insertAbove(this._elements.background);

	//helper function to format grid value
	var formatGridValue = function (value) {
		return UnitsConverter.toDisplay(this._appConfig.getUnits(), value, true);
	}.bind(this);

	//helper function to format time value
	var formatNumber = function (num) {
		return String(parseFloat(num.toFixed(2)));
	};

	var xCoord, yCoord;
	var timePerCell = (this._timeSpan / HistoryCanvas.GRID_NUM_CELLS_TIME) / 1000;

	//time axis
	for (var i = 1; i < HistoryCanvas.GRID_NUM_CELLS_TIME; i++) {
		xCoord = Math.round((this.VIEW_WIDTH / HistoryCanvas.GRID_NUM_CELLS_TIME) * i) + 0.5;
		gridGroup.addChild(new paper.Path({
			segments : [
					[xCoord, 0],
					[xCoord, this.VIEW_HEIGHT]
				],
			strokeColor : HistoryCanvas.GRID_COLOR
		}));

		//text
		gridGroup.addChild(new paper.PointText({
			point : [xCoord + 5, this.VIEW_HEIGHT - 10],
			content : formatNumber(i * timePerCell),
			fillColor : HistoryCanvas.GRID_TEXT_COLOR,
			justification : 'left'
		}));
	}

	// values axis
	var gridCellHeight = ((valsRange.to - valsRange.from) / HistoryCanvas.GRID_NUM_CELLS_VALS);
	for (var i = 1; i < HistoryCanvas.GRID_NUM_CELLS_VALS; i++) {
		yCoord = Math.round((this.VIEW_HEIGHT / HistoryCanvas.GRID_NUM_CELLS_VALS) * i) + 0.5;

		gridGroup.addChild(new paper.Path({
			segments : [
					[0, yCoord],
					[this.VIEW_WIDTH, yCoord]
				],
			strokeColor : HistoryCanvas.GRID_COLOR
		}));

		if (i > 1) {
			gridGroup.addChild(new paper.PointText({
				point : [5, this.VIEW_HEIGHT - yCoord - 10],
				content : formatGridValue(valsRange.from + i * gridCellHeight),
				fillColor : HistoryCanvas.GRID_TEXT_COLOR,
				justification : 'left'			
			}));
		}
	}

	//axises lines in corner		
	gridGroup.addChild(new paper.Path({
		segments : [
			[10.5, this.VIEW_HEIGHT - 30],
			[10.5, this.VIEW_HEIGHT - 10.5],
			[30, this.VIEW_HEIGHT - 10.5]
		],
		strokeColor : HistoryCanvas.GRID_TEXT_COLOR
	}));

	//axises labels
	gridGroup.addChild(new paper.PointText({
		point : [35, this.VIEW_HEIGHT - 7.5],
		content : 'TIME',
		fillColor : HistoryCanvas.GRID_TEXT_COLOR,
		justification : 'left'
	}));
	gridGroup.addChild(new paper.PointText({
		point : [15, this.VIEW_HEIGHT - 35],
		rotation : -90,
		justification : 'left',
		content : 'VALUE',
		fillColor : HistoryCanvas.GRID_TEXT_COLOR
	}));

	return gridGroup;
};


HistoryCanvas.prototype._getPointHoverCaption = function (point) {
	var caption;

	caption = UnitsConverter.toDisplay(this._units, point.distance, true);
	caption += ', ' + UnitsConverter.toDisplay(this._units, point.height, true);

	return '[' + caption + ']';
};

HistoryCanvas.prototype._drawPoints = function (valsRange, data, nowTimestamp) {
	var pointsGroup = new paper.Group();

	if (!data || !data.length){
		return pointsGroup;
	}

	var pixelsPerMicronValuesAxis = this.VIEW_HEIGHT / (valsRange.to - valsRange.from);

	// path init
	var pointsPath = new paper.Path({
		strokeColor : '#ff0000',
		strokeWidth : 2,
		strokeJoin : 'round',
		opacity : 1
	});
	pointsGroup.addChild(pointsPath);

	var pixelsPerMillisecond = this.VIEW_WIDTH / this._timeSpan;
	var minDataDistance = this._getMinDataDistance(data);

	//should draw circles as points ? - max points distance have to be 5 pixels
	var displayImagePoints = this._timeSpan <= 10000; //true;// (minDataDistance * pixelsPerMillisecond) > 5;
	if (displayImagePoints){
		var tmpCanvas = document.createElement('canvas');
		tmpCanvas.setAttribute('width', this.VIEW_WIDTH);
		tmpCanvas.setAttribute('height', this.VIEW_HEIGHT);
		var tmpCanvasContext = tmpCanvas.getContext('2d');
	}
	
	var tmpX, tmpY, tmpPath;

	for (var i = 0; i < data.length; i++) {

		tmpX = (nowTimestamp - data[i].timestamp) * pixelsPerMillisecond;
		// if there is gap larger than 200ms between measurements
		// start new line, leave gap in graph
		if (i > 0 && (data[i-1].timestamp - data[i].timestamp) > 200){
			// console.log('gap');
			tmpPath = new paper.Path({
				strokeColor : '#ff0000',
				strokeWidth : 2,
				strokeJoin : 'round',
				opacity : 1
			});
			pointsGroup.addChild(tmpPath);
			pointsPath = tmpPath;
		}
		// snapping of first point to beginning of graph
		// snap only if the measurement is not older than 100ms
		if (i == 0 && nowTimestamp - data[i].timestamp < 100){
			tmpX = 0;
		}

		tmpY = this.VIEW_HEIGHT - Math.round((data[i].data - valsRange.from) * pixelsPerMicronValuesAxis);
		pointsPath.addSegments([new paper.Point(tmpX, tmpY)]);
		
		//draw point circles
		if (displayImagePoints){
			tmpCanvasContext.beginPath();
			tmpCanvasContext.arc(tmpX, tmpY, HistoryCanvas.POINTS_CIRCLE_RADIUS, 0, 2 * Math.PI);
			tmpCanvasContext.fillStyle = '#ff0000';
			tmpCanvasContext.fill();
			tmpCanvasContext.closePath();
		}
		
	}


	//if drawing points, set data to canvas
	//otherwise simplify paths
	if (displayImagePoints){
		var canvasData = tmpCanvasContext.getImageData(0, 0, this.VIEW_WIDTH, this.VIEW_HEIGHT);
		this._elements.pointsImage.setSize(this.VIEW_WIDTH, this.VIEW_HEIGHT);
		this._elements.pointsImage.setImageData(canvasData);
		this._elements.pointsImage.setBounds(new paper.Rectangle(0,0,this.VIEW_WIDTH, this.VIEW_HEIGHT));
		this._elements.pointsImage.visible = true;
	}else{
		this._elements.pointsImage.visible = false;
	}

	return pointsGroup;
};


HistoryCanvas.prototype._redrawCursor = function (valsRange) {
	this._elements.cursor.removeChildren();

	var pixelPerTime = this.VIEW_WIDTH / this._timeSpan;
	var pixelsPerMicronValuesAxis = this.VIEW_HEIGHT / (valsRange.to - valsRange.from);

	//cursor
	this._elements.cursor.addChildren([
		new paper.Path({
			segments : [
					[0, 0],
					[HistoryCanvas.CURSOR_SIZE / 2, HistoryCanvas.CURSOR_SIZE],
					[HistoryCanvas.CURSOR_SIZE, 0],
				],
			position : new paper.Point([this._cursorPosition, HistoryCanvas.CURSOR_SIZE / 2]),
			fillColor : HistoryCanvas.CURSOR_COLOR,
			name : HistoryCanvas.ELEMENT_CURSOR
		}),
		new paper.Path.Line({
			from : [this._cursorPosition, 0],
			to : [this._cursorPosition, this.VIEW_HEIGHT],
			strokeColor : HistoryCanvas.CURSOR_COLOR,
			opacity : 0.8
		}),
	]);	
	
	//intersection
	if (this._intersection){	
		// console.log(
		// 	'draw',
		// 	this._intersection.timestamp * pixelPerTime,
		// 	this.VIEW_HEIGHT - ((this._intersection.data - valsRange.from) * pixelsPerMicronValuesAxis)
		// );

		this._elements.cursor.addChild(
			new paper.Path.Circle({
				center : [
					this._intersection.timestamp * pixelPerTime,
					this.VIEW_HEIGHT - ((this._intersection.data - valsRange.from) * pixelsPerMicronValuesAxis)
				],
				radius: 6,
				strokeColor : HistoryCanvas.CURSOR_COLOR,
				opacity : 0.7,
			})
		);
	}

	this._elements.cursor.bringToFront();

};

HistoryCanvas.prototype._removeOldPoints = function (nowTimestamp) {
	this._data = this._data.filter(function (points) {
		return (nowTimestamp - points.timestamp) < this._timeSpan;
	}.bind(this));
};

HistoryCanvas.prototype._getMinDataDistance = function (data) {
	var minDistance = Number.MAX_SAFE_INTEGER;
	for (var i = 1; i < data.length; i++) {
		if ((data[i-1].timestamp - data[i].timestamp) < minDistance){
			minDistance = data[i-1].timestamp - data[i].timestamp;
		}
	}
	return minDistance;
};


HistoryCanvas.prototype._onFrame = function (event) {
	this._frameCounter++;
	if (this._frameCounter % 2 !== 0) {
		return;
	}

	var now = Date.now();
	var data = this._data;

	//if in pause, don't remove old points and draw with timestamp of pause event
	if (this._pause){
		now = this._pause.now;
		data = this._pause.data;
	}else{
		this._removeOldPoints(now);
	}

	//compute values range
	var valsRange = this._getValuesZoomRange(data);

	this._updateCursorIntersections(data, now);

	//redraw grid
	this._elements.grid.removeChildren();
	this._elements.grid.remove();
	this._elements.grid = this._drawGrid(valsRange);

	if (this._elements.zoomRectangleSelection){
		this._elements.zoomRectangleSelection.bringToFront();
	}

	//redraw points
	this._elements.points.removeChildren();
	this._elements.points.remove();
	this._elements.points = this._drawPoints(valsRange, data, now);

	//cursor position
	this._elements.cursorPositionText.content = HistoryCanvas.CURSOR_POSITION_TEXT + (this._intersection ? ((this._intersection.timestamp / 1000).toFixed(3) + ' s') : '-')

	this._redrawCursor(valsRange);

	//fps text redraw
	// this._elements.numberOfSamplesValue.content = this._getSampleRateText();
};



module.exports = HistoryCanvas;
