

var React = require('react');
var ReactDOM = require('react-dom');
var SensorStore = require('../stores/SensorStore');
var UnitsConverter = require('../helpers/UnitsConverter');
var TabConstants = require('../constants/TabConstants');
var AppConfig = require('../objects/AppConfig');
var SensorActions = require('../actions/SensorActions');
var StreamCanvas = require('../objects/StreamCanvas');
var debounce = require('debounce');
var classNames = require('classnames');

var PIXEL_SIZE = 0.4375;

var CANVAS_STREAM = 'canvasStream';

/**
 * Canvas sizes
 * @type {Number}
 */
var TARGET_WIDTH = 640;
var TARGET_HEIGHT = 320;

var streamStopTimer = null;


var MeasureTab = React.createClass({
	REF_BTN_STREAM : 'btnStream',
	REF_BTN_STREAM_STOP : 'btnStreamStop',
	REF_BTN_STREAM_NO_DATA : 'btnStreamNoData',

	getInitialState : function(){
		return {
			streamData : null,
			avgMean : null,
			avgMin : null,
			avgMax : null,
			validCnt : null,
			zeroCnt : null,
			errCnt : null,
			aux : null,

			streamRunning : false,
			streamStoppping : false,
			streamStopTimerInterval : 3

		}
	},

	componentDidMount: function() {
		SensorStore.addConfigChangedListener(this._configChangedHandler);
		SensorStore.addStreamAvgReceivedListener(this._streamAvgReceivedHandler);
		SensorStore.addStreamAvgNoDataReceivedListener(this._streamAvgNoDataReceivedHandler);
		SensorStore.addStreamStartedListener(this._streamStartedHandler);
		SensorStore.addStreamStoppedListener(this._streamStoppedHandler);
		SensorStore.addTabChangedListener(this._tabChangeHandler);

		this._canvas = new StreamCanvas(this.refs[CANVAS_STREAM]);

		//debounce handler and add it to window
		this._refreshCanvasDebounced = debounce(this._refreshCanvas, 200);
		window.addEventListener('resize', this._refreshCanvasDebounced);

	},

	componentWillUnmount: function() {
		SensorStore.removeConfigChangedListener(this._configChangedHandler);
		SensorStore.removeStreamAvgReceivedListener(this._streamAvgReceivedHandler);
		SensorStore.removeStreamAvgNoDataReceivedListener(this._streamAvgNoDataReceivedHandler);
		SensorStore.removeStreamStartedListener(this._streamStartedHandler);
		SensorStore.removeStreamStoppedListener(this._streamStoppedHandler);
		SensorStore.removeTabChangedListener(this._tabChangeHandler);
		
		window.removeEventListener('resize', this._refreshCanvasDebounced);
	},

	_tabChangeHandler: function(){
		this._refreshCanvas();
	},

	_refreshCanvas : function() {
		//resize to wrapper size
		var wrapper = ReactDOM.findDOMNode(this.refs[CANVAS_STREAM]).parentNode.getBoundingClientRect();

		if (!wrapper.width){
			return;
		}

		//change params of canvas and variables depending on it
		var w = parseInt(wrapper.width);
		var h = parseInt((w / TARGET_WIDTH) * TARGET_HEIGHT);

		//force update with canvas setup as callback
		this._canvas.refresh(w, h);
	},

	_configChangedHandler : function(config){
		
	},


	_onBtnStreamStartClick : function (event) {
		SensorActions.streamAvgStart();
	},

	_onBtnStreamNoDataStartClick : function (event) {
		SensorActions.streamAvgNoDataStart();
	},

	_onBtnStreamStopClick : function (event) {
		this.setState({
			streamStopping : true
		}, function () {
			SensorActions.streamStop();
		});
	},

	_streamAvgReceivedHandler : function (data) {
		var nextState = {};

		this.setState({
			streamData : data
		}, function () {
			var rawData = this.state.streamData.slice(0, data.length - 13 - ((data.length - 13) % 3)) //minus 13 bytes and ensure divisibility by 3
			this._canvas.setStreamData(
				this._handleRawData(rawData)
			);
		}.bind(this));
	},

	_handleRawData : function (rawData) {
		var results = [];
		for (var i = 0; i < rawData.length / 3 - 1; i++) {
			results.push((rawData[3 * i] * 256 + rawData[3 * i + 1]));
		}
		return results;
	},

	_streamAvgNoDataReceivedHandler : function (resultsData) {
		this.setState({
			avgMean : (resultsData[0] * 256 + resultsData[1]),
			avgMin : (resultsData[2] * 256 + resultsData[3]),
			avgMax : (resultsData[4] * 256 + resultsData[5]),
			validCnt : resultsData[6] * 256 + resultsData[7],
			zeroCnt : resultsData[8] * 256 + resultsData[9],
			errCnt : resultsData[10] * 256 + resultsData[11],
			aux : resultsData[12],
		});
	},

	_streamStartedHandler : function () {
		this._canvas.resetData();

		this._resetStreamResults();
		this.setState({
			streamRunning : true,
			streamStopping : false
		});
		streamStopTimer = setTimeout(function () {
			this._onBtnStreamStopClick();
		}.bind(this), this.state.streamStopTimerInterval * 1000);
	},
	
	_streamStoppedHandler : function () {
		this.setState({
			streamRunning : false,
			streamStopping : false
		});	

		clearTimeout(streamStopTimer);

		// if stream with data
		if (this.state.streamData){
			var resultsData = this.state.streamData.slice(this.state.streamData.length - 13)

			this.setState({
				avgMean : (resultsData[0] * 256 + resultsData[1]),
				avgMin : (resultsData[2] * 256 + resultsData[3]),
				avgMax : (resultsData[4] * 256 + resultsData[5]),
				validCnt : resultsData[6] * 256 + resultsData[7],
				zeroCnt : resultsData[8] * 256 + resultsData[9],
				errCnt : resultsData[10] * 256 + resultsData[11],
				aux : resultsData[12],
			}, function () {
				this._canvas.setAvgMin(this.state.avgMin);
				this._canvas.setAvgMax(this.state.avgMax);
				this._canvas.setAvgMean(this.state.avgMean);
				this._canvas.setStreamData(this._handleRawData(
					this.state.streamData.slice(0, this.state.streamData.length - 13)
				));
			}.bind(this));
		}
	},

	_resetStreamResults : function () {
		this.setState({
			streamData : null,
			avgMean : null,
			avgMin : null,
			avgMax : null,
			validCnt : null,
			zeroCnt : null,
			errCnt : null,
			aux : null,
			streamData : null
		});
	},


	render: function() {

		var btnStreamClasses = classNames(
			'ui small button',
			{
				'disabled' : this.state.streamRunning
			}
		);
		var btnStreamNoDataClasses = classNames(
			'ui small button',
			{
				'disabled' : this.state.streamRunning
			}
		);
		var btnStreamStopClasses = classNames(
			'ui small button',
			{
				'disabled' : !this.state.streamRunning || this.state.streamStoppping,
				'loading' : this.state.streamStopping
			}
		);

		return (
			<div className="ui grid">
				<div className="row">
					<div className="sixteen wide column">
						<div className="ui form">
							<div className="inline field">
								<div
									ref={this.REF_BTN_STREAM}
									className={btnStreamClasses}
									onClick={this._onBtnStreamStartClick}
									style={{marginRight : '0.25em'}}
								>
									Stream
								</div>
								<div
									ref={this.REF_BTN_STREAM_NO_DATA}
									className={btnStreamNoDataClasses}
									onClick={this._onBtnStreamNoDataStartClick}
								>
									Stream (no data)
								</div>
								<div
									ref={this.REF_BTN_STREAM_STOP}
									className={btnStreamStopClasses}
									onClick={this._onBtnStreamStopClick}
								>
									Stop
								</div>
								&nbsp;&nbsp;
								<label>Stop stream timer [s]</label>
								&nbsp;
								<input
									type="number"
									min="1"
									max="60"
									className="ui input"
									value={this.state.streamStopTimerInterval}
									onChange={function (event) {
										this.setState({streamStopTimerInterval : event.currentTarget.value});
									}.bind(this)}
								/>
							</div>
						</div>
					</div>
				</div>
				<div className="row">
					<div className="sixteen wide column">
						<div className="ui grid">
							<div className="two wide column">
								<div className="ui mini statistic">
									<div className="label" style={{whiteSpace : 'nowrap'}}>
										Avg. min
									</div>
									<div className="value">
										{(this.state.avgMin === null) ? '-' : Math.round(this.state.avgMin * PIXEL_SIZE)}
									</div>
								</div>
							</div>
								{/*<label>Avg. min</label><br />
								{(this.state.avgMin === null) ? '-' : this.state.avgMin}*/}
							<div className="two wide column">
								<div className="ui mini statistic">
									<div className="label" style={{whiteSpace : 'nowrap'}}>
										Avg. mean
									</div>
									<div className="value">
										{(this.state.avgMean === null) ? '-' : Math.round(this.state.avgMean * PIXEL_SIZE)}
									</div>
								</div>
							</div>
								{/*<label>Avg. mean</label><br />
								{(this.state.avgMean === null) ? '-' : this.state.avgMean}*/}
							<div className="two wide column">
								<div className="ui mini statistic">
									<div className="label" style={{whiteSpace : 'nowrap'}}>
										Avg. max
									</div>
									<div className="value">
										{(this.state.avgMax === null) ? '-' : Math.round(this.state.avgMax * PIXEL_SIZE)}
									</div>
								</div>
							</div>
								{/*<label>Avg. max</label><br />
								{(this.state.avgMax === null) ? '-' : this.state.avgMax}*/}
							<div className="two wide column">
								<div className="ui mini statistic">
									<div className="label" style={{whiteSpace : 'nowrap'}}>
										Valid count
									</div>
									<div className="value">
										{(this.state.validCnt === null) ? '-' : this.state.validCnt}
									</div>
								</div>
							</div>
								{/*<label>Valid count</label><br />
								{(this.state.validCnt === null) ? '-' : this.state.validCnt}*/}
							<div className="two wide column">
								<div className="ui mini statistic">
									<div className="label" style={{whiteSpace : 'nowrap'}}>
										Zero count
									</div>
									<div className="value">
										{(this.state.zeroCnt === null) ? '-' : this.state.zeroCnt}
									</div>
								</div>
							</div>
								{/*<label>Zero count</label><br />
								{(this.state.zeroCnt === null) ? '-' : this.state.zeroCnt}*/}
							<div className="two wide column">
								<div className="ui mini statistic">
									<div className="label" style={{whiteSpace : 'nowrap'}}>
										Error count
									</div>
									<div className="value">
										{(this.state.errCnt === null) ? '-' : this.state.errCnt}
									</div>
								</div>
							</div>
								{/*<label>Error count</label><br />
								{(this.state.errCnt === null) ? '-' : this.state.errCnt}*/}
						</div>
					</div>
		        </div>
		        <div className="row">
					<div className="sixteen wide column">
						<div className="canvas-wrapper" id="streamWrapper" ref="streamWrapper">
							<canvas
								id={CANVAS_STREAM}
								ref={CANVAS_STREAM}
								data-target-width={TARGET_WIDTH}
								data-target-height={TARGET_HEIGHT}
								data-paper-keepalive="true"
							></canvas>
    					</div>
    				</div>
    			</div>
    		</div>
	    );
	}

});

module.exports = MeasureTab;