
var React = require('react');
var ReactDOM = require('react-dom');
var SensorStore = require('../stores/SensorStore');
var SensorActions = require('../actions/SensorActions');
var SensorConfig = require('./../objects/sensor/sensorConfig');
var classNames = require('classnames');
var debounce = require('debounce');


/**
 * Time to debounce validation
 * @type {Number}
 */
var DEBOUNCE_TIME = 500;

var REF_ROOT = 'sensorParams';

var SensorParams = React.createClass({
	propTypes : {
		disabled : React.PropTypes.bool.isRequired
	},

	getInitialState: function() {
		return this._getEmptyState();
	},

	componentDidMount: function() {
		SensorStore.addConfigChangedListener(this._configChangedHandler);

		this.debounceValidateParams = debounce(this.debounceValidateParams, DEBOUNCE_TIME);

		$(ReactDOM.findDOMNode(this.refs[REF_ROOT])).find('.item').tab();
	},

	componentWillUnmount: function() {
		SensorStore.removeConfigChangedListener(this._configChangedHandler);
	},
	
	_configChangedHandler : function(config){
		this._updateStateFromConfig(config);
	},

	_getEmptyState : function(){
		return {
			mode : null,
			threshold : null,
		};
	},

	_updateStateFromConfig : function(config){
		var newState = {	
			mode : config.getMode(),
			threshold : config.getThreshold(),
		};
		
		this._updateState(newState);
	},

	_updateState : function(newState){
		this.setState(newState);

		//debounced validate with marking errors
		this.debounceValidateParams(true);
	},


	_sensorParamsChangeHandler : function (event) {
		var stateValue = event.currentTarget.getAttribute('data-sensor-action');
		var newState = {};

		newState[stateValue] = event.currentTarget.value;
		if (event.currentTarget.type == "checkbox"){
			newState[stateValue] = event.currentTarget.checked;
		}

		this._updateState(newState);
	},

	_checkInputValidity : function(input) {
		if (input.type === 'checkbox') {
			return true;
		}

		var min = input.getAttribute('min');
		var max = input.getAttribute('max');

		if (!min || !max)  {
			throw "Input limits not set";
		}

		min = parseInt(min);
		max = parseInt(max);

		if (input.value >= min && input.value <= max){
			if (input.getAttribute('data-even-only') !== 'true' || input.value % 2 === 0){
				return true;
			}
		}

		return false;
	},


	getConfig : function () {
		if (!this.immediateValidateParams(false)){
			return null;
		};

		var config = new SensorConfig();			
		
		config.setMode(this.state.mode);
		config.setThreshold(this.state.threshold);
		
		return config;
	},

	/**
	 * Input validation for debouncing - with indicating error inputs
	 */
	debounceValidateParams : function () {
		this.immediateValidateParams(true);
	},

	/**
	 * Immediate input validation with optional indication of error inputs
	 */
	immediateValidateParams : function (indicateErrors) {
		if (this.props.disabled)
			return true;

		var input,
		    valid = true,
		    params = (ReactDOM.findDOMNode(this.refs[REF_ROOT])) ? ReactDOM.findDOMNode(this.refs[REF_ROOT]).querySelectorAll('input') : [];

		for (var i = 0; i < params.length; i++) {
			input = params[i];
			if (!this._checkInputValidity(input)){
				valid = false;
				if (indicateErrors) {
					input.parentNode.classList.add('error');
				}
				//console.log('Invalid input:', input);
			}else{
				input.parentNode.classList.remove('error');
			}
		};

		return valid;
	},

	render: function() {
		var elementClasses = classNames(
			'sensor-params',
			{
				'disabled' : this.props.disabled
			}
		);

		return (
			<div
				className={elementClasses}
				ref={REF_ROOT}
			>
				<div className="ui bottom attached active segment tab" data-tab="params-image">
					<div className="ui grid">
						<div className="six wide column">
							<div className="ui small form">
								<div className="ui inline field">
									<label className="w160">Mode</label>
									<select
										className="w90"
										data-sensor-action="mode" 
										onChange={this._sensorParamsChangeHandler}
										value={this.state.mode === null ? '' : this.state.mode}
										disabled={this.props.disabled}
									>
										<option value="0">Edge 1</option>
										<option value="1">Edge 2</option>
										<option value="2">Dia</option>
										<option value="3">Gap</option>
										<option value="4">Center</option>
										<option value="5">Solid edge</option>
									</select>
								</div>
								<div className="ui inline field">
									<label className="w160">Threshold</label>
									<input
										className="w90"
										data-sensor-action="threshold" 
										onChange={this._sensorParamsChangeHandler}
										value={this.state.threshold === null ? '' : this.state.threshold}
										disabled={this.props.disabled}
										type="number"
										min="0"
										max="2048"
									/>
								</div>
							
								
							</div>
						</div>						
					</div>
				</div>
				
			</div>
	    );
	}

});

module.exports = SensorParams;
