var fs = nw.require('fs');
var React = require('react');
var ReactDOM = require('react-dom');
var SensorStore = require('../stores/SensorStore');
var SensorActions = require('../actions/SensorActions');
var AppStore = require('../stores/AppStore');
var TabConstants = require('../constants/TabConstants');
var GlobalConstants = require('../constants/GlobalConstants');
var UnitsConverter = require('../helpers/UnitsConverter');
var FilenameGenerator = require('../helpers/FilenameGenerator');
var AppConfig = require('../objects/AppConfig');
var ProfileCanvas = require('../objects/ProfileCanvas');
var debounce = require('debounce');
var assign = require('object-assign');

var CANVAS_PROFILE = 'canvasProfile';

/**
 * Canvas sizes
 * @type {Number}
 */
var TARGET_WIDTH = 640;
var TARGET_HEIGHT = 512;

/**
 * Meauring frames per second
 * @type {Number}
 */
var fpsCounter = 0; //current counter of data
var fpsTime = performance.now(); //measuring time between frames

var ProfileTab = React.createClass({
	propTypes : {
		pause : React.PropTypes.bool.isRequired,
		onTresholdChange : React.PropTypes.func.isRequired
	},

	getInitialState : function(){
		return {
			profileRaw : null,
			profileNormalizedFiltered : null,
			profileTemplate : null,
		};
	},

	componentDidMount: function() {
		SensorStore.addProfileReceivedListener(this._getProfileHandler);
		SensorStore.addTabChangedListener(this._tabChangeHandler);
		SensorStore.addConfigChangedListener(this._configChangedHandler);

		this._canvas = new ProfileCanvas(this.refs[CANVAS_PROFILE]);
		this._canvas.on(ProfileCanvas.EVENT_THRESHOLD_CHANGE, this.props.onTresholdChange);

		//debounce handler and add it to window
		this._refreshCanvasDebounced = debounce(this._refreshCanvas, 200);
		window.addEventListener('resize', this._refreshCanvasDebounced);
	},

	componentWillUnmount: function() {
		SensorStore.removeProfileReceivedListener(this._getProfileHandler);
		SensorStore.removeTabChangedListener(this._tabChangeHandler);
		SensorStore.removeConfigChangedListener(this._configChangedHandler);

		window.removeEventListener('resize', this._refreshCanvasDebounced);
	},

	_refreshCanvas : function() {
		//resize to wrapper size
		var wrapper = ReactDOM.findDOMNode(this.refs[CANVAS_PROFILE]).parentNode.getBoundingClientRect();

		if (!wrapper.width){
			return;
		}

		//change params of canvas and variables depending on it
		var w = parseInt(wrapper.width);
		var h = parseInt((w / TARGET_WIDTH) * TARGET_HEIGHT);

		//force update with canvas setup as callback
		this._canvas.refresh(w, h);
	},

	_tabChangeHandler: function(tab){
		if (!this.props.pause) {
			//redraw empty canvas
			this._canvas.setProfileRaw(null);
			this._canvas.setProfileNormalized(null);
			this._canvas.setProfileNormalizedFiltered(null);
			this._refreshCanvas();
		}else{
			this._refreshCanvas();
		}
	},

	_configChangedHandler : function(config){
		if (typeof config.getThreshold() == "number"){
			this._canvas.setThreshold(config.getThreshold());
		}

		if (config.getActiveLines()){
			this._canvas.setActiveLines(config.getActiveLines());
		}
	},

	_getProfileHandler : function(profile){
		if (this.props.pause){
			return;
		}

		var nextState = {};
		if (profile.type == GlobalConstants.PROFILE_CMOS){
			nextState.profileRaw = profile.data;
			this._canvas.setProfileRaw(profile.data);
		}
		if (profile.type == GlobalConstants.PROFILE_NORMALIZED_FILTERED){
			nextState.profileNormalizedFiltered = profile.data;
			this._canvas.setProfileNormalizedFiltered(profile.data);
		}

		this.setState(nextState);
		this._updateFpsOnProfileReceive();
	},

	_updateFpsOnProfileReceive : function () {
		fpsCounter++;

		//update fps ?
		var tmpTime = performance.now();
		if (tmpTime - fpsTime > 2000){
			this._canvas.setFps(parseInt(fpsCounter / 2));
			fpsCounter = 0;
			fpsTime = tmpTime;
		}		
	},

	_profileTemplateFromCsv : function (filePath, callback) {


		fs.readFile(
			filePath,
			'utf8',
			function (err, csvContent) {
				if (err){
					return null;
				}

				var output = [];
				var lines = csvContent.split(require('os').EOL);

				lines.forEach(function (val) {
					console.log(val);
					output.push(parseFloat(val));
				});

				callback(output);
			}
		);		
	},

	_saveFile : function (filename, data) {


		fs.writeFile(
			filename,
			data,
			'ascii',
			function(err) {
				if(err) {
					alertify.error('Couldn\'t save file');
				}else{
					alertify.success('File saved');
				}
			}
		);
	},

	_getProfileCsv : function (data) {
		var newLine = require('os').EOL;
		return data.join(newLine);
	},

	saveCsv : function () {
		var toSave = []
		var newLine = require('os').EOL;

		if (this.state.profileNormalizedFiltered && this.state.profileNormalizedFiltered.length){
			this._saveFile(
				FilenameGenerator.profile('csv'),
				this._getProfileCsv(this.state.profileNormalizedFiltered)
			);
		}		
	},

	savePng : function () {
		this._saveFile(
			FilenameGenerator.profile('png'),
			this._canvas.getPng()
		);	
	},

	setTemplate : function(filePath) {
		var self = this;

		//set template from file if argument was passed
		if (filePath) {
			this._profileTemplateFromCsv(
				filePath,
				function(templateData) {
					self.setState({
						profileTemplate : templateData
					}, function () {
						this._canvas.setTemplate(this.state.profileTemplate);
					}.bind(this));
				}.bind(this)
			);
		}else{
			//else set current profile as template
			this.setState({
				profileTemplate : this.state.profileNormalizedFiltered
			}, function () {
				this._canvas.setTemplate(this.state.profileTemplate);
			}.bind(this));
		}	
	},

	cancelTemplate : function() {
		this.setState({
			profileTemplate : null
		}, function () {
			this._canvas.cancelTemplate();
		}.bind(this));
	},

	saveTemplate : function() {
		var toSave = this.state.profileTemplate ? this.state.profileTemplate : this.state.profileNormalizedFiltered;
		this._saveFile(
			FilenameGenerator.templateProfile('csv'),
			this._getProfileCsv(toSave)
		);
	},

	switchZoom : function (isOn) {
		if (isOn) {
			this._canvas.setZoom();
		}else{
			this._canvas.cancelZoom();
		}
	},

	zoomPlus : function () {
		this._canvas.zoomPlus();
	},

	zoomMinus : function () {
		this._canvas.zoomMinus();
	},

	showRawProfile : function () {
		this._canvas.showRawProfile();
	},

	hideRawProfile : function () {
		this._canvas.hideRawProfile();
	},


	render: function() {
		return (
			<div className="canvas-wrapper" id="profileWrapper" ref="profileWrapper">
				<canvas
					id={CANVAS_PROFILE}
					ref={CANVAS_PROFILE}
					data-target-width={TARGET_WIDTH}
					data-target-height={TARGET_HEIGHT}
					data-paper-keepalive="true"
				></canvas>
			</div>
		);
	}

});

module.exports = ProfileTab;