
var React = require('react');
var ReactDOM = require('react-dom');
var SensorStore = require('../stores/SensorStore');
var ModeButtonSwitcher = require('./ModeButtonSwitcher.jsx');
var UnitsConverter = require('../helpers/UnitsConverter');
var AppConfig = require('../objects/AppConfig');
var HistoryCanvas = require('../objects/HistoryCanvas');
var debounce = require('debounce');
var classNames = require('classnames');


// timer for continuous updating measuredValue in state - avoid fast flickering rerendering of value
var measuredValueUpdateTimer;

/**
 * Canvas names constants
 */
var CANVAS_HISTORY = 'history';


/**
 * Canvas sizes
 * @type {Number}
 */
var TARGET_WIDTH = 640;
var TARGET_HEIGHT = 300;


var MeasureTab = React.createClass({
	TIME_SPAN_MIN : 2,
	TIME_SPAN_MAX : 60,
	REFRESH_INTERVAL : 50,
	measResultAllModes : 0,

	getInitialState : function(){
		return {
			measuredValue : 0,
			mode : 0,

			cursorValue : 0,
			cursorPosition : 0,

			timeSpan : 5,

			pause : false,
			valuesAutoZoom : true
		}
	},

	componentDidMount: function() {
		SensorStore.addConfigChangedListener(this._configChangedHandler);
		SensorStore.addMeasureDataReceivedListener(this._dataReceived);
		SensorStore.addTabChangedListener(this._tabChangeHandler);

		this._canvas = new HistoryCanvas(this.refs[CANVAS_HISTORY], AppConfig);
		this._canvas.setTimeSpan(this.state.timeSpan);
		this._canvas.on(HistoryCanvas.EVENT_CURSOR_POINTS, (data) => {
			var nextState = {
				cursorValue : null,
				cursorPosition : null
			}
			if (data){
				nextState.cursorValue = data.data;
				nextState.cursorPosition = data.timestamp;
			}
			this.setState(nextState);
		});

		//debounce handler and add it to window
		this._refreshCanvasDebounced = debounce(this._refreshCanvas, 200);
		window.addEventListener('resize', this._refreshCanvasDebounced);

		//timer to update measured value - avoid fast flickering
		measuredValueUpdateTimer = setInterval(function () {
			if (!this.state.pause){
				this.setState({
					measuredValue : this.measResultAllModes[this.state.mode] ? this.measResultAllModes[this.state.mode] : 0
				});
			}
		}.bind(this), this.REFRESH_INTERVAL);

		//debounced set time span - avoid fast changes when writing with keyboard
		//also validate input
		this._debounceSetTimeSpan = debounce(function () {
			var setTimeSpan = function () {
				this._canvas.setTimeSpan(this.state.timeSpan);	
			}.bind(this);

			if (this.state.timeSpan < this.TIME_SPAN_MIN){
				this.setState({
					timeSpan : this.TIME_SPAN_MIN
				}, setTimeSpan);
			}else if (this.state.timeSpan > this.TIME_SPAN_MAX){
				this.setState({
					timeSpan : this.TIME_SPAN_MAX
				}, setTimeSpan);
			}else{
				setTimeSpan();
			}
			
		}.bind(this), 200);
	},

	componentWillUnmount: function() {
		SensorStore.removeConfigChangedListener(this._configChangedHandler);
		SensorStore.removeMeasureDataReceivedListener(this._dataReceived);
		SensorStore.removeTabChangedListener(this._tabChangeHandler);

		window.removeEventListener('resize', this._refreshCanvasDebounced);

		clearInterval(measuredValueUpdateTimer);
	},

	_tabChangeHandler: function(tab){
		this._refreshCanvas();
	},

	_refreshCanvas : function() {
		//resize to wrapper size
		var wrapper = ReactDOM.findDOMNode(this.refs[CANVAS_HISTORY]).parentNode.getBoundingClientRect();

		if (!wrapper.width){
			return;
		}

		//change params of canvas and variables depending on it
		var w = parseInt(wrapper.width);
		var h = parseInt((w / TARGET_WIDTH) * TARGET_HEIGHT);

		//force update with canvas setup as callback
		this._canvas.refresh(w, h);
	},

	// update class property, state is updated with timer
	_dataReceived : function (data) {
		// console.log(data);
		
		this.measResultAllModes = data
		this._canvas.addData(this.measResultAllModes[this.state.mode]);
	},


	_configChangedHandler : function(config){
		if (typeof config.getMode() === 'number'){
			this.setState({
				mode : parseInt(config.getMode())
			});
		}
	},

	_resetDataClick : function () {
		this._canvas.resetData();
	},

	_onTimeSpanChange : function (event) {
		this.setState({
			timeSpan : event.currentTarget.value
		}, function () {
			this._debounceSetTimeSpan(this.state.timeSpan);
		}.bind(this));		
	},

	_onPauseClick : function (event) {
		this.setState({
			pause : !this.state.pause
		}, function () {
			if (this.state.pause){
				this._canvas.setPause();
			}else{
				this._canvas.resetPause();
			}
		}.bind(this));	
	},

	_onValuesAutoZoomClick : function (event) {
		this.setState({
			valuesAutoZoom : !this.state.valuesAutoZoom
		}, () => {
			if (this.state.valuesAutoZoom){
				this._canvas.enableValuesAutoZoom();
			}else{
				this._canvas.disableValuesAutoZoom();
			}
		});
	},

	render: function() {
		var pauseButtonClasses = classNames(
			'ui button',
			{
				'green' : this.state.pause
			}
		);
		var valuesZoomClasses = classNames(
			'ui small icon button',
			{
				'green' : this.state.valuesAutoZoom,
			}
		);
		return (
			<div className="ui grid">
				<div className="row">
					<div className="eight wide column">
					<div className="ui grid">
							<div className="sixteen wide column" style={{paddingBottom : 0}}>
								<ModeButtonSwitcher />
							</div>
							<div className="sixteen wide column">
								<div className="ui small form">
									<div className="inline field">
										<div className="ui buttons">
											<div
												className={pauseButtonClasses}
												onClick={this._onPauseClick}
											>
												Pause
											</div>
											<div
												className="ui button"
												onClick={this._resetDataClick}
											>
												Reset
											</div>
											<div
												className={valuesZoomClasses}
												onClick={this._onValuesAutoZoomClick}
											>
												Auto zoom
											</div>
										</div>
										
										<label  style={{marginLeft:'20px'}}>
											Time span [s]
										</label>
										&nbsp;
										<input
											type="number"
											min={this.TIME_SPAN_MIN}
											max={this.TIME_SPAN_MAX}
											value={this.state.timeSpan}
											onChange={this._onTimeSpanChange}
										/>
										
										
									</div>
								</div>
							</div>
						</div>
					</div>
					<div className="eight wide column">
						<div className="ui grid" >
							<div className="twelve wide column" style={{paddingBottom : 0}} >
								<div className="ui header" style={{fontSize : '60px'}} >
									{UnitsConverter.toDisplay(AppConfig.getUnits(), this.state.measuredValue)}
								</div>
							</div>
							<div className="seven wide column" style={{paddingTop : 0}}>
								<div className="ui header" >
									Cursor: {(this.state.cursorPosition == null) ? '-' : (UnitsConverter.toDisplay(AppConfig.getUnits(), this.state.cursorValue))}
								</div>
							</div>
							{/*<div className="seven wide column" style={{paddingTop : 0}}>
								<div className="ui header" >
									Position: {(this.state.cursorPosition == null) ? '-' : ((this.state.cursorPosition / 1000).toFixed(3)) + ' s'}
								</div>
							</div>*/}
						</div>
					</div>
				</div>
				<div className="row">
					<div className="sixteen wide column">
						<div className="canvas-wrapper" id="streamWrapper" ref="streamWrapper">
							<canvas
								id={CANVAS_HISTORY}
								ref={CANVAS_HISTORY}
								data-target-width={TARGET_WIDTH}
								data-target-height={TARGET_HEIGHT}
								data-paper-keepalive="true"
							></canvas>
    					</div>
    				</div>
    			</div>
			
    		</div>
	    );
	}

});

module.exports = MeasureTab;